"""
Author:            Arpad Kiss, sekter@mail.matav.hu
Modified:          1999.06.21.
Platform:          -
Description:       TreeBox version 0.79
                   With this widget you can load items by '<<FillLevel>>' event or by AddNode method.
                   This widget is static. You cannot change the behaviour of it. You can query 
		   the init parameters, but you cannot set them.
                   If a branch has been loaded by <<FillLevel>> then it won't be refreshed during the lifetime of the 
		   widget, except you call the Refresh method which deletes the subitems of the 
		   given item, and loads subitems just under the given item if this is in expanded
		   state.
                   You can Expand, Collapse and Select an item giving its key. 
                   Value returns the text of the node.
                   This widget may be used by mouse or keyboard.
		   Events: '<<FillLevel>>','<<BeforIconUpdate>>','<<BeforeTextUpdate>>','<<NodeClick-1>>','<<NodeClick-3>>'
                   All variables/functions begining with an underscore are considered as local. Don't set them from
                   outside of the class.
Todos:             The space between items depends on the font size, so you can get overlapped 
		   items if the height of your icons or +/- signs is greater then the font.?NodeHeight?
		   tabspace?!
		   RemoveNode implementation is ugly
		   if it is not mapped then AddNode shouldn't draw lines, icons etc.
"""


#Imports
#************************************************************************************************************
from Tkinter import *
from Tkinter import _cnfmerge
#************************************************************************************************************



#Constants
#************************************************************************************************************
_iexpanded_data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xf8, 0x1, 0xac, 0x2, 0x54, 0xfd, 0xac, 0xaa, 0x54, 0xd5, 0xff, 0xbf, 0x1, 0xe0, 0x1, 0xa0, 0x2, 0xc0, 0x2, 0xc0, 0x4, 0x80, 0xfc, 0xff};'
_iexpanded_maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xf8, 0x1, 0xfc, 0x3, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xf8, 0xff};'

_icollapsed_data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xf8, 0x1, 0x4, 0x2, 0x54, 0xfd, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0xfc, 0xff};'
_icollapsed_maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xf8, 0x1, 0xfc, 0x3, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff};'

_ileaf_data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xfc, 0x3f, 0x4, 0x20, 0xf4, 0x2f, 0x4, 0x20, 0xf4, 0x2f, 0x4, 0x20, 0xf4, 0x2f, 0x4, 0x20, 0xf4, 0x2f, 0x4, 0x20, 0xfc, 0x3f, 0x0, 0x0};'
_ileaf_maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0x0, 0x0};'

posCHILD=0
posFIRST=1
posPREVIOUS=2
posNEXT=3
posLAST=4
#************************************************************************************************************



#
#************************************************************************************************************
class TreeBox(Canvas):
    
    def __init__(self,
                 root=None,
		 font='helvetica 10',
		 textcolor='black',
		 signsize=4,
		 signgap=24,
		 signcolor='gray40',
		 linecolor='gray50',
		 visiblelines=YES,
		 visiblesigns=YES,
		 visibleicons=NO,
		 icongap=4,
		 linespace=2,
		 selcolor='yellow',
		 icons={'expanded'  : {'data' : _iexpanded_data, 'maskdata' : _iexpanded_maskdata,    'background' : 'yellow', 'foreground' : 'gray40'},
		        'collapsed' : {'data' : _icollapsed_data,'maskdata' : _icollapsed_maskdata,   'background' : 'yellow', 'foreground' : 'gray40'},
			'leaf'      : {'data' : _ileaf_data,     'maskdata' : _ileaf_maskdata,        'background' : 'white',  'foreground' : 'gray40'}},
		 **kw):
	#
	Canvas.__init__(self,root,kw)
	self._topx,self._topy=5,15
	self._SelectedNode=None 
	self._icons={'expanded'  : BitmapImage(data=_iexpanded_data,maskdata=_iexpanded_maskdata,background='yellow',foreground='gray40'),
	             'collapsed' : BitmapImage(data=_icollapsed_data,maskdata=_icollapsed_maskdata,background='yellow',foreground='gray40'),
		     'leaf'      : BitmapImage(data=_ileaf_data,maskdata=_ileaf_maskdata,background='white',foreground='gray40')}
	self._improot={'CanvasItems': [], # canvasitem ids related to this node
	            'VerticalLines': [],
		    'Children': [],	 # subnodes list
		    'NextSibling': None, 
		    'PreviousSibling' : None,
		    'MainCanvasID': None,# this the id of the text
		    'Parent': None,      # parent node
		    'Expanded': TRUE,
		    'Key': None,
		    'Text': "",
		    'Font': font,
		    'Textcolor': textcolor,
		    'Icon': self._icons['leaf']
		    } # implicit root
	self._Nodes={} #{ user defined node key : node dict(similar to self._improot) } 
	self._Node_ids={} # {canvas id: node dict in self._Nodes } ; multiple ids(text, icon, sign) point to the same node dict, although text handled as a main canvas item
	self._scrollbox_width=19
	#
	self._font=font
	self._textcolor=textcolor
	self._signsize=signsize
	self._signgap=signgap
	self._signcolor=signcolor
	self._linecolor=linecolor
	self._visiblelines=visiblelines
	self._visiblesigns=visiblesigns
	self._visibleicons=visibleicons
	self._icongap=icongap
	self._tab=signsize*2+signgap
	self._linespace=linespace
	self._selcolor=selcolor
	#Scrollbars
	self._vbar=Scrollbar(self)
	self['yscrollcommand']=self._vbar.set
	self._vbar['command']=self.yview
	self._hbar=Scrollbar(self,orient=HORIZONTAL)
	self['xscrollcommand']=self._hbar.set
	self._hbar['command']=self.xview
	self._scrollbox_width=int(self._vbar.cget('width'))+int(self._vbar.cget('borderwidth'))*2
	#bottom_right corner
	self._brcorner=Frame(self,relief=RAISED,borderwidth=2)
	#bindings
	self.bind("<KeyPress-Return>",self._TreeBox_KeyPress_Return)
	self.bind("<KeyPress-Home>",self._TreeBox_KeyPress_Home)
	self.bind("<KeyPress-End>",self._TreeBox_KeyPress_End)
	self.bind("<KeyPress-Down>",self._TreeBox_KeyPress_Down)
	self.bind("<KeyPress-Up>",self._TreeBox_KeyPress_Up)
	self.bind("<KeyPress-Left>",self._TreeBox_KeyPress_Left)
	self.bind("<KeyPress-Right>",self._TreeBox_KeyPress_Right)
	self.bind("<KeyPress-Next>",self._TreeBox_KeyPress_PgDown)
	self.bind("<KeyPress-Prior>",self._TreeBox_KeyPress_PgUp)
	self.bind("<ButtonPress>",self._TreeBox_Click)
	self.bind("<Double-1>",self._TreeBox_DoubleClick)
	self.bind("<Configure>",self._Configure)
	self.bind("<Map>",self._Map)
	
    def __getitem__(self,i):
	return self.cget(i)
		       
    def configure(self,cnf={},**kw):
	if kw:
		kw = _cnfmerge((cnf,kw))
	else:
		kw=_cnfmerge(cnf)
	self._PreConfig(kw)
	Canvas.configure(self,kw)
	
    def cget(self,parKey):
	if parKey=='Value':	
	    try:    
		return self._SelectedNode['text']
	    except:
	        return None
	elif parKey=='SelectedNode':	    
	    try:
		return self._SelectedNode
	    except:
	        return None
	else:
	    try:
		return eval('self._'+parKey)
	    except:
		# this is not mine, call the parent
		return Canvas.cget(self,parKey)
    
    #some properties have to been processed and  deleted, then the others are sent to Canvas
    def _PreConfig(self,kw):
	if kw.has_key('font'):
	    del kw['font']
	if kw.has_key('textcolor'):
	    del kw['textcolor']
	if kw.has_key('signsize'):
	    del kw['signsize']
	if kw.has_key('signgap'):
	    del kw['signgap']
	if kw.has_key('signcolor'):
	    del kw['signcolor']
	if kw.has_key('linecolor'):
	    del kw['linecolor']
	if kw.has_key('visiblelines'):
	    del kw['visiblelines']
	if kw.has_key('visiblesigns'):
	    del kw['visiblesigns']
	if kw.has_key('visibleicons'):
	    del kw['visibleicons']
	if kw.has_key('icongap'):
	    del kw['icongap']
	if kw.has_key('linespace'):
	    del kw['linespace']
	if kw.has_key('selcolor'):
	    del kw['selcolor']
	if kw.has_key('icons'):
	    del kw['icons']
    
    # '<Map>' event-handler	
    def _Map(self,event):
	# it is needed, because before mainloop the events are not handled
	# so the icons are not updated after AddNode
	self._Collapse(self._improot)
	self._Expand(self._improot)
    
    # '<Configure>' event-handler	
    def _Configure(self,event):
	self._SetScrolling()

    def _TreeBox_Click(self,event):
	self.focus_set()
	ids=self.find_withtag(CURRENT)
	if len(ids)>0:
	    ItemID=ids[0]
	    self._node=self._Node_ids[ItemID]
	    self.event_generate('<<NodeClick-%i>>' % event.num)
	    tags=self.gettags(ItemID)
	    if 'nodetext' in tags:
		self._Select(self._Node_ids[ItemID])
	    elif 'nodesign' in tags:
		if self._Node_ids[ItemID]['Expanded']:
		    self._Collapse(self._Node_ids[ItemID])
		else:
		    self._Expand(self._Node_ids[ItemID])
	    elif 'nodeicon' in tags:
		self._Select(self._Node_ids[ItemID])

    def _TreeBox_KeyPress_Up(self,event):
	#up
	if self._SelectedNode:
	    self._Select(self._GetPrevious(self._SelectedNode))		
		
    def _TreeBox_KeyPress_Down(self,event):
	#down
	self._Select(self._GetNext())
		
    def _TreeBox_KeyPress_Left(self,event):
	#left
	if self._hbar.place_info():
	    self.xview('scroll',-1,'units')
		
    def _TreeBox_KeyPress_Right(self,event):
	#right
	if self._hbar.place_info():
	    self.xview('scroll',1,'units')
		
    def _TreeBox_KeyPress_PgUp(self,event):
	#pageup
	if self._vbar.place_info():
	    if self._hbar.place_info():
		ps=self._scrollbox_width+1
	    else:
		ps=0
	    try:
		y=self.bbox(self._SelectedNode['MainCanvasID'])[1]-float(self.winfo_height()-ps)*9/10
	    except:
		return
	    if y<0:
		ids=[]
		try:
		    ids.append(self._improot['Children'][0])
		except:
		    return
	    else:
		ids=self.find_overlapping(0,y,self.bbox(ALL)[3]+self._scrollbox_width,y+self._linespace)
	    if ids:
		self._Select(self._Node_ids[ids[0]])
		
    def _TreeBox_KeyPress_PgDown(self,event):
	#pagedown
	if self._vbar.place_info():
	    if self._hbar.place_info():
		ps=self._scrollbox_width+1
	    else:
		ps=0
	    try:
		y=self.bbox(self._SelectedNode['MainCanvasID'])[1]+float(self.winfo_height()-ps)*9/10
	    except:
		return
	    if y>self.bbox(ALL)[3]-self._linespace:
		ids=[]
		ids.append(self._GetLast())
	    else:
		ids=self.find_overlapping(0,y,self.bbox(ALL)[3]+self._scrollbox_width,y+self._linespace)
	    if ids:
		self._Select(self._Node_ids[ids[0]])

    def _TreeBox_KeyPress_Home(self,event):
	#home
	try:
	    self._Select(self._improot['Children'][0])
	except:
	    pass
		
    def _TreeBox_KeyPress_End(self,event):
	#end
	self._Select(self._GetLast())

    def _TreeBox_KeyPress_Return(self,event):
	#enter
	if self._SelectedNode:
	    if self._SelectedNode['Expanded']:
	    	self._Collapse(self._SelectedNode)
	    else:
	    	self._Expand(self._SelectedNode)

    def _TreeBox_DoubleClick(self,event):
	ids=self.find_withtag(CURRENT)
	if len(ids)>0:
	    tags=self.gettags(ids[0])
	    if 'nodetext' in tags:	    
		if self._Node_ids[ids[0]]['Expanded']:
		    #it is expanded
		    self._Collapse(self._Node_ids[ids[0]])
		else:
		    #it is collapsed
		    self._Expand(self._Node_ids[ids[0]])
	    elif 'nodeicon' in tags:
		if self._Node_ids[ids[0]]['Expanded']:
		    self._Collapse(self._Node_ids[ids[0]])
		else:
		    self._Expand(self._Node_ids[ids[0]])
    
    def FullPath(self,NodeKey):
	fp=[]
	try:
	    inode=self._Nodes[NodeKey]
	    while inode['Key']!=self._improot['Key']:
		fp.insert(0,inode['Key'])
		inode=inode['Parent']
	except:
	    fp=None
	return fp
    
    def Select(self,NodeKey):
	if not self._Nodes.has_key(NodeKey): return FALSE #bad NodeKey
	self._Select(self._Nodes[NodeKey])
	return TRUE
		    
    def _Select(self,parNode):
	# deleting the old selection
	if parNode:
	    if not parNode['MainCanvasID']:
		fp=self.FullPath(parNode['Key'])
		if fp:
		    for ikey in fp:
		    	if not self._Nodes[ikey]['Expanded']:
			    self.Expand(ikey)
	    ids=self.find_withtag('selected')
	    if len(ids)>0:
	    	oldselID=ids[0]
	    	self.delete(oldselID) # delete the box
	    	self._Node_ids[oldselID]['CanvasItems'].remove(oldselID) # delete this canvas item from canvas items of the node
	    	del self._Node_ids[oldselID] # delete this ref to node
	    # setting the new
	    x1,y1,x2,y2=self.bbox(parNode['MainCanvasID'])
	    selID=self.create_rectangle(x1,y1,x2+5,y2,fill=self._selcolor,tags=('selected','canvasitems'))
	    self._Node_ids[selID]=parNode
	    parNode['CanvasItems'].append(selID)
	    self.tkraise(parNode['MainCanvasID'],selID)
	    self._SelectedNode=parNode
	    #scroll to this item
	    up,vy=self.yview()
	    abb=self.bbox(ALL)
	    h=float(abb[3])+int(self._scrollbox_width)
	    if self._hbar.place_info():
	    	ps=self._scrollbox_width+1
	    else:
	    	ps=0
	    if up>float(y1-self._linespace)/h:
	    	self.yview('moveto',float(y1)/h)
	    elif self.winfo_height()<y2-h*up+2*int(self['borderwidth'])+3+ps:
	    	self.yview('moveto',float(y2-self.winfo_height()+2*int(self['borderwidth'])+3+ps)/h)
	    left,vx=self.xview()
	    w=float(abb[2])+int(self._scrollbox_width)
	    if left>float(x1)/w:
	    	self.xview('moveto',float(x1)/(w+int(self._scrollbox_width)))
	    elif vx<float(x1+int(self._scrollbox_width)-w*left)/(w-w*left):
	    	self.xview('moveto',float(x1)/(w+int(self._scrollbox_width)))
    
    # get the node just above the current in the canvas
    def _GetPrevious(self,parNode=None):
	if not parNode:
	    return None
	elif parNode['PreviousSibling']:
	    inode=parNode['PreviousSibling']
	    while inode['Expanded'] and inode['Children'] and inode['Children'][0]['MainCanvasID']:
		inode=inode['Children'][len(inode['Children'])-1]
	    return inode
	elif parNode['Parent']:
	    if parNode['Parent']['Key']==self._improot['Key']:
		return parNode
	    else:
		return parNode['Parent']
	else:
	    return parNode
	
    # get the node just under the current in the canvas
    # firstcall: if it is false thar means we come from a children and we don't want to go back there
    def _GetNext(self,parNode=None,firstcall=YES):
	if not parNode:
	    try:
		parNode=self._SelectedNode
	    except:
		return None
	if parNode['Expanded'] and firstcall:
	    try:
		return parNode['Children'][0]
	    except:
		pass
	if parNode['NextSibling']!=None:
	    return parNode['NextSibling']
	elif parNode['Parent'] and  parNode['Parent']['Key']!=self._improot['Key']:
	    return self._GetNext(parNode['Parent'],NO)
	else:
	    return parNode

    # get the last node in the canvas
    def _GetLast(self):
	if self._improot['Children']:
	    inode=self._improot['Children'][len(self._improot['Children'])-1]
	    while inode['Expanded']:
	    	inode=inode['Children'][len(inode['Children'])-1]
	    return inode

    def Collapse(self,NodeKey):
	try:
	    if self._Nodes[NodeKey]['Expanded']:
		self._Collapse(self._Nodes[NodeKey])
	    return TRUE
	except:
	    return FALSE
		
    #LeaveExpanded=TRUE when we call this function to hide the subbranches
    def _Collapse(self,parNode=None,LeaveExpanded=FALSE):
	if not parNode:
	    ids=self.find_withtag(CURRENT)
	    if len(ids)>0:
		parNode=self._Node_ids[ids[0]]
	if parNode and parNode['Key']!=self._improot['Key'] and not self.Visible(parNode['Key']):
	    if not LeaveExpanded:
		parNode['Expanded']=FALSE
	elif parNode and parNode['Key']!=self._improot['Key']:
	    MainCanvasID=parNode['MainCanvasID'] # CURRENT may be a sign
	    tmpc=self['cursor']
	    self['cursor']='watch'
	    # we add 'move' to items under MainCanvasID
	    self._SetMoveTag(parNode)
	    # deleting the subitems, if this has
	    # looping throught the subitems level
	    inodes=parNode['Children']
	    space=0
	    for inode in inodes:
		bb=self.bbox(inode['MainCanvasID'])
		space=space+abs(bb[3]-bb[1])+self._linespace
	    	if  inode['Expanded']:
		    #it is expanded, so we have to call this function again
		    self._Collapse(inode,TRUE)
	    	self._DeleteCanvasItems(inode)
		inode['MainCanvasID']=None
	    # we have to pull up the items with 'move' tag
	    self._MoveCanvasItems(parNode,-space)
	    if not LeaveExpanded:
	    	# set the expanded flag to FALSE
	    	self._SetExpanded(parNode,FALSE)
	    self['cursor']=tmpc
	elif parNode['Key']==self._improot['Key']:
	    #<<Map>> event uses it!
	    if self._Nodes:
		self.delete(ALL)
		for i in self._Nodes.items():
		    self._DeleteCanvasItems(i[1])
	    parNode['Expanded']=FALSE

    def Expand(self,NodeKey):
	try:
	    if not self._Nodes[NodeKey]['Expanded']:
		self._Expand(self._Nodes[NodeKey])
	    return TRUE
	except:
	    return FALSE
	
    def _Expand(self,parNode=None):
	if not parNode:
	    ids=self.find_withtag(CURRENT)
	    if len(ids)>0:
		parNode=self._Node_ids[ids[0]]
	if parNode:
	    if not self.Visible(parNode['Key']) and parNode['Key']!=self._improot['Key']:
		parNode['Expanded']=TRUE
		return
	    tmpc=self['cursor']
	    self['cursor']='watch'
	    space=0
	    MainCanvasID=parNode['MainCanvasID']
	    if MainCanvasID:
	    	# level 1,2,....
		bb=self.bbox(MainCanvasID)
            	x,y=bb[0]+self._tab,bb[3]+self._linespace
	    else:
	    	#this is the implicit root, level 0
	    	x,y=self._topx+self._tab,self._topy            
            if parNode['Children']:
	    	# it has subitems
	    	# we have to make room for the new items
            	# at first we add 'move' tag to items under MainCanvasID
	    	self._SetMoveTag(parNode)
	    	# creating the new text items	
	    	for inode in parNode['Children']:
		    newID=self.create_text(x,y,text=inode['Text'],anchor=NW)
		    x1,y1,x2,y2=self.bbox(newID)
		    self.move(newID,0,y-y1)
		    inode['MainCanvasID']=newID
		    self._Node_ids[newID]=inode
		    inode['CanvasItems']=[newID]
	            inode['VerticalLines']=[]
		    self._node=inode
		    self.event_generate('<<BeforeTextUpdate>>')
	            self.itemconfigure(newID,tags=('nodetext','canvasitems'),font=inode['Font'],fill=inode['Textcolor'])
	            self._DrawLines(inode)
	            self._UpdateSign(inode)
	            self._UpdateIcon(inode)
		    bb=self.bbox(newID)
		    y=bb[3]+self._linespace
		    space=space+abs(bb[3]-bb[1])+self._linespace
		    # refresh the rectangle of the selection
		    if self._SelectedNode and self._SelectedNode['Key']==inode['Key']:
		    	self._Select(self._SelectedNode)
	    	# we have to make room for the new items
	    	# we are moving items with 'move' tag
	    	self._MoveCanvasItems(parNode,space)
	    	# we set the expanded flag
		self._SetExpanded(parNode,TRUE)
 	    	# if these were expanded, then we have to expand them    
	    	for inode in parNode['Children']:
		    if not inode['Children']:
			self._node=inode
			self.event_generate('<<FillLevel>>')
		    if inode['Expanded']:
		    	#it is expanded, so we have to call this function again
			self._Expand(inode)
	    self['cursor']=tmpc

    def _SetMoveTag(self,parNode=None):
        if parNode['MainCanvasID']:
	    items=self.find_withtag('canvasitems')
	    iy2=self.bbox(parNode['MainCanvasID'])[3]	
	    # moving
	    for i in items:
		y2=self.bbox(i)[3]	
		if y2>iy2 and self._Node_ids[i]['Key']!=parNode['Key']:
		    self.addtag_withtag('move'+str(parNode['MainCanvasID']),i)
	    # redrawing vertical lines    
	    tmpNode=parNode
	    while 1:
		for iline in tmpNode['VerticalLines']:
		    x1,y1,x2,y2=self.coords(iline)
		    if y1>y2: y1,y2=y2,y1
		    if y2>iy2:
			self.addtag_withtag('redraw'+str(parNode['MainCanvasID']),iline)
		if tmpNode['Parent']['Key']==self._improot['Key']: break
		tmpNode=tmpNode['Parent']
	    

    # moves items set by _SetMoveTag
    def _MoveCanvasItems(self,parNode,space):
        if parNode and parNode['MainCanvasID']:
	    items=self.find_withtag('move'+str(parNode['MainCanvasID']))
	else:
	    items=[]
	# moving
	for i in items:
	    self.move(i,0,space)
	# redrawing vertical lines    
        if parNode and parNode['MainCanvasID']:
	    items=self.find_withtag('redraw'+str(parNode['MainCanvasID']))
	else:
	    items=[]
	for i in items:
	    x1,y1,x2,y2=self.coords(i)
	    if y1>y2: y1,y2=y2,y1
	    sc=(y2-y1+space)/(y2-y1)
	    self.scale(i,x2,y2,1,sc)
	# delete the 'move' and 'redraw' tag
        if parNode and parNode['MainCanvasID']:
	    self.dtag('move'+str(parNode['MainCanvasID']),'move'+str(parNode['MainCanvasID']))
	    self.dtag('redraw'+str(parNode['MainCanvasID']),'redraw'+str(parNode['MainCanvasID']))
	# set the scrolled region
	self._SetScrolling()

    def _DrawLines(self,parNode):
	if not self._visiblelines: return
	MainCanvasID=parNode['MainCanvasID']
	#creating the horizontal one
	bb=self.bbox(MainCanvasID)
	x,y=bb[0],bb[1]+(bb[3]-bb[1])/2
	if parNode['Children']:
	    newID=self.create_line(x-self._signgap,y,x,y,tags=('line','ihline','canvasitems'),fill=self._linecolor)
	else:
	    newID=self.create_line(x-self._signgap-self._signsize,y,x,y,tags=('line','ihline','canvasitems'),fill=self._linecolor)
	parNode['CanvasItems'].append(newID)
	self._Node_ids[newID]=parNode
	#creating the vertical one
	UpNode=self._GetPrevious(parNode)
	if UpNode['Key']!=parNode['Key']: 
	    # there is a item above this
	    x1=x-self._signgap-self._signsize
	    bb=self.bbox(UpNode['MainCanvasID'])
	    if parNode['Parent']['Key']==UpNode['Parent']['Key']:
	    	# they are on the same level
	    	if UpNode['Children']:
		    if parNode['Children']:
		    	newID=self.create_line(x1,y-self._signsize,x1,bb[1]+(bb[3]-bb[1])/2+self._signsize,tags=('line','ivline','canvasitems'),fill=self._linecolor)
		    else:
		    	newID=self.create_line(x1,y,x1,bb[1]+(bb[3]-bb[1])/2+self._signsize,tags=('line','ivline','canvasitems'),fill=self._linecolor)
	    	else:
		    if parNode['Children']:
		    	newID=self.create_line(x1,y-self._signsize,x1,bb[1]+(bb[3]-bb[1])/2,tags=('line','ivline','canvasitems'),fill=self._linecolor)
		    else:
		    	newID=self.create_line(x1,y,x1,bb[1]+(bb[3]-bb[1])/2,tags=('line','ivline','canvasitems'),fill=self._linecolor)
	    else:
	    	if parNode['Children']:
		    newID=self.create_line(x1,y-self._signsize,x1,bb[3],tags=('line','ivline','canvasitems'),fill=self._linecolor)
	    	else:
		    newID=self.create_line(x1,y,x1,bb[3],tags=('line','ivline','canvasitems'),fill=self._linecolor)
	    
	    parNode['CanvasItems'].append(newID)
	    self._Node_ids[newID]=parNode
	    UpNode['VerticalLines'].append(newID)

    def _UpdateSign(self,parNode):
	if not self._visiblesigns: return
	if not parNode['MainCanvasID']: return
	MainCanvasID=parNode['MainCanvasID']
	#deleting the old sign
	items=self.find_withtag('nodesign')
	for iItemID in items:
	    if self._Node_ids[iItemID]['MainCanvasID']==MainCanvasID:
		self.delete(iItemID)
		self._Node_ids[iItemID]['CanvasItems'].remove(iItemID)
		del self._Node_ids[iItemID]
	#creating new one
	bb=self.bbox(MainCanvasID)
	x,y=bb[0]-self._signgap-self._signsize,bb[1]+(bb[3]-bb[1])/2
	if parNode['Children']:
	    # box
	    newID=self.create_polygon(x-self._signsize,y-self._signsize,x+self._signsize,y-self._signsize,x+self._signsize,y+self._signsize,x-self._signsize,y+self._signsize,tags=('nodesign','canvasitems'),outline=self._signcolor,fill='')
	    parNode['CanvasItems'].append(newID)
	    self._Node_ids[newID]=parNode
	    if parNode['Expanded']:
		# - sign
		newID=self.create_line(x-self._signsize+2,y,x+self._signsize-1,y,tags=('nodesign','canvasitems'),fill=self._signcolor)
		parNode['CanvasItems'].append(newID)
		self._Node_ids[newID]=parNode
	    else:
		# + sign
		newID=self.create_line(x-self._signsize+2,y,x+self._signsize-1,y,tags=('nodesign','canvasitems'),fill=self._signcolor)
		parNode['CanvasItems'].append(newID)
		self._Node_ids[newID]=parNode
		newID=self.create_line(x,y-self._signsize+2,x,y+self._signsize-1,tags=('nodesign','canvasitems'),fill=self._signcolor)
		parNode['CanvasItems'].append(newID)
		self._Node_ids[newID]=parNode

    def _UpdateIcon(self,parNode):
	if not self._visibleicons: return
	self._node=parNode
	self.event_generate('<<BeforeIconUpdate>>')
	MainCanvasID=parNode['MainCanvasID']
	if MainCanvasID:
	    #deleting the old icon
	    items=self.find_withtag('nodeicon')
	    for iItemID in items:
	    	if self._Node_ids[iItemID]['MainCanvasID']==MainCanvasID:
		    self.delete(iItemID)
		    self._Node_ids[iItemID]['CanvasItems'].remove(iItemID)
		    del self._Node_ids[iItemID]
	    #creating new one
	    bb=self.bbox(MainCanvasID)
	    x,y=bb[0]-self._icongap,bb[1]+(bb[3]-bb[1])/2
	    newID=self.create_image(x,y,image=parNode['Icon'],tags=('nodeicon','canvasitems'),anchor=E)
	    parNode['CanvasItems'].append(newID)
	    self._Node_ids[newID]=parNode
	
    def _SetScrolling(self):
	abb=self.bbox(ALL)
	if abb:
	    self['scrollregion']='0 0 '+str(abb[2]+int(self._scrollbox_width))+' '+str(abb[3]+int(self._scrollbox_width))
	    if (self.yview()[1]<1 or self.yview()[0]!=0) and (self.xview()[1]<1 or self.xview()[0]!=0):
	    	self._vbar.place(relx=1,x=-int(self['borderwidth']),rely=0,y=int(self['borderwidth']),relheight=1,height=-2*int(self['borderwidth'])-self._scrollbox_width,anchor=NE)
	    	self._vbar.set(self.yview()[0],self.yview()[1])
	    	self._hbar.place(relx=0,x=int(self['borderwidth']),rely=1,y=-int(self['borderwidth']),relwidth=1,width=-2*int(self['borderwidth'])-self._scrollbox_width,anchor=SW)
	    	self._hbar.set(self.xview()[0],self.xview()[1])
	    	self._brcorner.place(relx=1,x=-int(self['borderwidth']),rely=1,y=-int(self['borderwidth']),width=self._scrollbox_width,height=self._scrollbox_width,anchor=SE)
	    elif (self.yview()[1]<1 or self.yview()[0]!=0) and self.xview()[1]==1 and self.xview()[0]==0:
	    	self._vbar.place(relx=1,x=-int(self['borderwidth']),rely=0,y=int(self['borderwidth']),relheight=1,height=-2*int(self['borderwidth']),anchor=NE)
	    	self._vbar.set(self.yview()[0],self.yview()[1])
	    	self._hbar.place_forget()
	    	self._brcorner.place_forget()
	    elif self.yview()[1]==1 and self.yview()[0]==0 and (self.xview()[1]<1 or self.xview()[0]!=0):
	    	self._vbar.place_forget()
	    	self._hbar.place(relx=0,x=int(self['borderwidth']),rely=1,y=-int(self['borderwidth']),relwidth=1,width=-2*int(self['borderwidth']),anchor=SW)
	    	self._hbar.set(self.xview()[0],self.xview()[1])
	    	self._brcorner.place_forget()
	    else:
	    	self._vbar.place_forget()
	    	self._hbar.place_forget()
	    	self._brcorner.place_forget()
	
    # called by _Collapse
    def _DeleteCanvasItems(self,parNode):
	MainCanvasID=parNode['MainCanvasID']
	for i in parNode['CanvasItems']:
	    self.delete(i)
	    del self._Node_ids[i]
	    if parNode['Parent']['Key']!=None and i in parNode['Parent']['VerticalLines']:
		parNode['Parent']['VerticalLines'].remove(i)
	parNode['CanvasItems']=[]
	parNode['VerticalLines']=[]

    def _SetExpanded(self,parNode,Value):
	#if parNode!=self._improot:
	parNode['Expanded']=Value
	self._node=parNode
	self._UpdateSign(parNode)
	self._UpdateIcon(parNode)

    def Refresh(self,NodeKey):
	tmpc=self['cursor']
	n=self._Nodes[NodeKey]
	self['cursor']='watch'
	itwasexpanded=NO
	if n['Expanded']:
	    self._Collapse(n)
	    itwasexpanded=YES
	self._DelChildNodes(n)
	self._node=n
	self.event_generate('<<FillLevel>>')
	if itwasexpanded:
	    self._Expand(n)
	self['cursor']=tmpc

    def _DelChildNodes(self,parNode):
	if parNode['Expanded'] and self.Visible(parNode['Key']):
	    self._Collapse(parNode)
	for iid in parNode['Children']:
	    self._DelChildNodes(iid)
	    del self._Nodes[iid['Key']]
	    if iid['Key']==self._SelectedNode['Key']:
		self._SelectedNode=parNode
	parNode['Children']=[]

    def RemoveNode(self,NodeKey):
	n=self._Nodes[NodeKey]
	p=n['Parent']
	itwasexpanded=p['Expanded']
	self._Collapse(p)
	self._DelChildNodes(n)
	p['Children'].remove(n)
	del self._Nodes[NodeKey]
	if n['PreviousSibling'] and n['NextSibling']:
	    n['PreviousSibling']['NextSibling']=n['NextSibling']
	    n['NextSibling']['PreviousSibling']=n['PreviousSibling']
	elif n['PreviousSibling']:
	    n['PreviousSibling']['NextSibling']=None
	elif n['NextSibling']:
	    n['NextSibling']['PreviousSibling']=None
	if itwasexpanded:
	    self._Expand(p)
	if n==self._SelectedNode:
	    self._SelectedNode=p
	
    def AddNode(self,NodeKey,NodeText,RelativeKey=None,Position=posCHILD):
	node={'CanvasItems': [], # canvasitem ids related to this node
	      'VerticalLines': [],
	      'Children': [],	 # subnodes list
	      'NextSibling': None, 
	      'PreviousSibling' : None,
	      'MainCanvasID': None,# this the id of the text
	      'Parent': None,      # parent node
	      'Expanded': FALSE,
	      'Key': NodeKey,
	      'Text': NodeText,
	      'Font': self._font,
	      'Textcolor': self._textcolor,
	      'Icon': self._icons['leaf']
	      }
	if RelativeKey: 
	    if Position==posCHILD:
		node['Parent']=self._Nodes[RelativeKey]
		if node['Parent']['Children']:
		    node['Parent']['Children'][len(node['Parent']['Children'])-1]['NextSibling']=node
		    node['PreviousSibling']=node['Parent']['Children'][len(node['Parent']['Children'])-1]
		node['Parent']['Children'].append(node)
	    elif Position==posFIRST:
		node['Parent']=self._Nodes[RelativeKey]['Parent']
		if node['Parent']['Children']:
		    node['Parent']['Children'][0]['PreviousSibling']=node
		    node['NextSibling']=node['Parent']['Children'][0]
		node['Parent']['Children'].insert(0,node)
	    elif Position==posPREVIOUS:
		node['Parent']=self._Nodes[RelativeKey]['Parent']
		if node['Parent']['Children']:
		    node['PreviousSibling']=self._Nodes[RelativeKey]['PreviousSibling']
		    node['NextSibling']=self._Nodes[RelativeKey]
		    if self._Nodes[RelativeKey]['PreviousSibling']:
			self._Nodes[RelativeKey]['PreviousSibling']['NextSibling']=node
		    self._Nodes[RelativeKey]['PreviousSibling']=node
		node['Parent']['Children'].insert(node['Parent']['Children'].index(self._Nodes[RelativeKey]),node)
	    elif Position==posNEXT:
		node['Parent']=self._Nodes[RelativeKey]['Parent']
		if node['Parent']['Children']:
		    node['PreviousSibling']=self._Nodes[RelativeKey]
		    node['NextSibling']=self._Nodes[RelativeKey]['NextSibling']
		    self._Nodes[RelativeKey]['NextSibling']['PreviousSibling']=node
		    self._Nodes[RelativeKey]['NextSibling']=node
		node['Parent']['Children'].insert(node['Parent']['Children'].index(self._Nodes[RelativeKey])+1,node)
	    elif Position==posLAST:
		node['Parent']=self._Nodes[RelativeKey]['Parent']
		if node['Parent']['Children']:
		    node['Parent']['Children'][len(node['Parent']['Children'])-1]['NextSibling']=node
		    node['PreviousSibling']=node['Parent']['Children'][len(node['Parent']['Children'])-1]
		node['Parent']['Children'].append(node)
	else:
	    # only posCHILD is allowed
	    if Position!=posCHILD:
		raise ValueError
	    node['Parent']=self._improot
	    if node['Parent']['Children']:
		node['Parent']['Children'][len(node['Parent']['Children'])-1]['NextSibling']=node
		node['PreviousSibling']=node['Parent']['Children'][len(node['Parent']['Children'])-1]
	    node['Parent']['Children'].append(node)
	self._Nodes[NodeKey]=node
	if self.Visible(node['Key']):
	    #we have to draw the canvas items
	    prevnode=self._GetPrevious(node)
	    MainCanvasID=prevnode['MainCanvasID']
	    if MainCanvasID:
		# level 1,2,....
		x,y=self._topx+self._tab,self.bbox(MainCanvasID)[3]+self._linespace
	    else:
		#this is the implicit root, level 0
		x,y=self._topx+self._tab,self._topy
	    # we have to make room for the new item
            # at first we add 'move' tag to items under MainCanvasID
	    self._SetMoveTag(prevnode)
	    # creating the new text item
	    newID=self.create_text(x,y,text=node['Text'],anchor=NW)
	    x1,y1,x2,y2=self.bbox(newID)
	    self.move(newID,0,y-y1)
	    node['MainCanvasID']=newID
	    self._Node_ids[newID]=node
	    node['CanvasItems']=[newID]
	    node['VerticalLines']=[]
	    self._node=node
	    self.event_generate('<<BeforeTextUpdate>>')
	    self.itemconfigure(newID,tags=('nodetext','canvasitems'),font=node['Font'],fill=node['Textcolor'])
	    self._DrawLines(node)
	    self._UpdateSign(node)
	    self._UpdateIcon(node)
	    bb=self.bbox(newID)
	    y=bb[3]+self._linespace
	    space=abs(bb[3]-bb[1])+self._linespace
	    # we have to make room for the new items
	    # we are moving items with 'move' tag
	    self._MoveCanvasItems(prevnode,space)
	if node['Parent']['MainCanvasID']:
	    self._node=node['Parent']
	    self._UpdateSign(node['Parent'])
	    self._UpdateIcon(node['Parent'])
	    
    def Visible(self,NodeKey):
	if NodeKey:
	    inode= self._Nodes[NodeKey]['Parent']
	    if inode:
	    	while inode and inode['Expanded']:
		    inode=inode['Parent']
	    	return inode==None
	    else:
		return FALSE
	else:
	    return FALSE
	    
    def Clear(self):
	self.delete(ALL)
	self._Nodes={}
	self._Node_ids={}
	self._improot['Children']=[]
	self._SelectedNode=None
	self._improot['Expanded']=YES
	
    def FillTreeBox(self,NodeKey=None):
	tmpc=self['cursor']
	self['cursor']='watch'
	self.Clear()
	self._improot['Key']=NodeKey
	self._node=self._improot
	self.event_generate('<<FillLevel>>')
	for inode in self._improot['Children']:
	    self._node=inode
	    self.event_generate('<<FillLevel>>')
	self['cursor']=tmpc
	    
#************************************************************************************************************



#Test
#************************************************************************************************************
def test():
    
    root=Tk()
    
    #icons
    icons={}
    data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x88, 0x8, 0x32, 0x15, 0x44, 0x55, 0xea, 0x32, 0x90, 0xa, 0xf8, 0x3e, 0xc6, 0x47, 0xb1, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
    maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xd8, 0xa, 0xfe, 0x37, 0x76, 0x7d, 0xfe, 0xbf, 0xbf, 0x6b, 0xff, 0xfe, 0xff, 0x7f, 0xbe, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
    icons['treeinb']= BitmapImage(data=data,maskdata=maskdata,background='pink',foreground='brown')
    data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x88, 0x8, 0x32, 0x15, 0x44, 0x55, 0xea, 0x32, 0x90, 0xa, 0xf8, 0x3e, 0xc6, 0x47, 0xb1, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
    maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xd8, 0xa, 0xfe, 0x37, 0x76, 0x7d, 0xfe, 0xbf, 0xbf, 0x6b, 0xff, 0xfe, 0xff, 0x7f, 0xbe, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
    icons['tree']= BitmapImage(data=data,maskdata=maskdata,background='green',foreground='brown')
    data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x80, 0x0, 0x20, 0x0, 0x20, 0x0, 0x90, 0xf, 0x48, 0x12, 0x28, 0x22, 0xfc, 0x7f, 0x12, 0x42, 0x19, 0x99, 0xef, 0xf7, 0x34, 0x2c, 0x18, 0x18};'
    maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x80, 0x0, 0x20, 0x0, 0x20, 0x0, 0x90, 0xf, 0x48, 0x12, 0x28, 0x22, 0xfc, 0x7f, 0xfe, 0x7f, 0xff, 0xff, 0xff, 0xff, 0x3c, 0x3c, 0x18, 0x18};'
    icons['car']= BitmapImage(data=data,maskdata=maskdata,background='red',foreground='black')
    icons['redleaf']= BitmapImage(data=_ileaf_data,maskdata=_ileaf_maskdata,background='red',foreground='black')

    
    # treeview filled by AddNode method
    lbl=Label(root,text='Filled by AddNode method',relief=GROOVE)
    lbl.place(relx=0,rely=0,relwidth=0.5,relheight=0.1)
    trb1=TreeBox(root,visibleicons=YES,bg='orange',linecolor='blue',selcolor='lightgreen',signcolor='red')
    trb1.place(relx=0,rely=0.1,relwidth=0.5,relheight=0.8)

    def Select(tb=trb1):
	if tb.cget('SelectedNode'):
	    tb.Select(tb.cget('SelectedNode')['Key'])
    def Expand(tb=trb1):
	if tb.cget('SelectedNode'):
	    tb.Expand(tb.cget('SelectedNode')['Key'])
    def Collapse(tb=trb1):
	if tb.cget('SelectedNode'):
	    tb.Collapse(tb.cget('SelectedNode')['Key'])

	    
    def BeforeTextUpdate(event=None):
	if event.widget._node['Key']=='l11' or event.widget._node['Parent']['Key']=='l11':
	    if event.widget._node['Key']=='l11':
		event.widget._node['Textcolor']='darkgreen'
		event.widget._node['Font']='helvetica 16 bold'
	    elif event.widget._node['Key'][-1:]=='1':
		event.widget._node['Textcolor']='darkgreen'
		event.widget._node['Font']='helvetica 14 bold italic'
	    else:
		event.widget._node['Textcolor']='darkgreen'
		event.widget._node['Font']='helvetica 8 bold'
    def BeforeIconUpdate(event=None,icons=icons):
	if event.widget._node['Expanded']:
	    if event.widget._node['Key']=='l11':
		event.widget._node['Icon']=icons['treeinb']
	    elif event.widget._node['Key']=='l23':
		event.widget._node['Icon']=icons['car']
	    else:
		event.widget._node['Icon']=event.widget._icons['expanded']
	elif event.widget._node['Children']:
	    if event.widget._node['Key']=='l11':		
		event.widget._node['Icon']=icons['tree']
	    elif event.widget._node['Key']=='l23':
		event.widget._node['Icon']=icons['car']
	    else:
		event.widget._node['Icon']=event.widget._icons['collapsed']
	else:
	    if event.widget._node['Parent']['Key']=='l11':
		event.widget._node['Icon']=icons['tree']
	    elif event.widget._node['Parent']['Key']=='l23':
		event.widget._node['Icon']=icons['redleaf']
	    else:
		event.widget._node['Icon']=event.widget._icons['leaf']
		
    def NodeClick1(event=None):
	if event.widget._SelectedNode:
	    print "'<<NodeClick-1>>' event,selected: %s,clicked: %s" % (event.widget._SelectedNode['Text'],event.widget._node['Text'])
	else:
	    print "'<<NodeClick-1>>' event,selected: None,clicked: %s" % event.widget._node['Text']
    def NodeClick3(event=None):
	from tkMessageBox import showinfo
	showinfo(title="'<<NodeClick-3>>' event", message='%s' % event.widget._SelectedNode['Text'])
    
    trb1.bind('<<BeforeIconUpdate>>',BeforeIconUpdate)
    trb1.bind('<<BeforeTextUpdate>>',BeforeTextUpdate)
    trb1.bind('<<NodeClick-1>>',NodeClick1)
    trb1.bind('<<NodeClick-3>>',NodeClick3)

    trb1.AddNode('l11','Trees')
    trb1.AddNode('l21','Oak','l11',posCHILD)
    trb1.AddNode('l22','Chestnut','l21',posPREVIOUS)
    trb1.AddNode('l12','Animals')
    trb1.AddNode('l13','Things')
    trb1.AddNode('l23','Cars','l13')
    trb1.AddNode('l24','Pencil','l13')
    trb1.AddNode('l25','Hammer','l23',posLAST)
    trb1.AddNode('l26','Table','l24',posFIRST)
    trb1.AddNode('l31','Ford','l23')
    trb1.AddNode('l32','Chevrolet','l23')
    
    btn=Button(root,text='Show selected',command=Select)
    btn.place(x=0,rely=1,relwidth=0.17,relheight=0.1,anchor=SW)
    btn=Button(root,text='Expand',command=Expand)
    btn.place(relx=0.17,rely=1,relwidth=0.17,relheight=0.1,anchor=SW)
    btn=Button(root,text='Collapse',command=Collapse)
    btn.place(relx=0.34,rely=1,relwidth=0.16,relheight=0.1,anchor=SW)


    # treeview filled by FillTreeBox method(that triggers the FillLevel event)
    lbl=Label(root,text='Fill button calls FillTreeBox method',relief=GROOVE)
    lbl.place(relx=0.5,rely=0,relwidth=0.5,relheight=0.1)
    trb2=TreeBox(root,visibleicons=YES)
    trb2.place(relx=0.5,rely=0.1,relwidth=0.5,relheight=0.8)

    def FillLevel(event=None):
	if event.widget._node['Key']==None:
	    event.widget.AddNode('l11','Trees')
	    event.widget.AddNode('l12','Animals')
	    event.widget.AddNode('l13','Things')
	elif event.widget._node['Key']=='l11':
	    event.widget.AddNode('l21','Oak','l11',posCHILD)
	    event.widget.AddNode('l22','Chestnut','l11',posCHILD)
	elif event.widget._node['Key']=='l13':
	    event.widget.AddNode('l23','Cars','l13')
	    event.widget.AddNode('l24','Pencil','l13')
	    event.widget.AddNode('l25','Hammer','l13')
	    event.widget.AddNode('l26','Table','l13')
	elif event.widget._node['Key']=='l23':
	    event.widget.AddNode('l31','Ford','l23')
	    event.widget.AddNode('l32','Chevrolet','l23')
	else:
	    pass

    def BeforeIconUpdate2(event=None):
	if event.widget._node['Children']:
	    if event.widget._node['Expanded']:
		event.widget._node['Icon']=event.widget._icons['expanded']
	    else:
		event.widget._node['Icon']=event.widget._icons['collapsed']

    def NodeClick32(event=None):
	from tkMessageBox import showinfo
	showinfo(title="'<<NodeClick-3>>' event", message='%s' % len(event.widget._SelectedNode['Children']))
    
    def Refresh(tb=trb2):
	if tb['SelectedNode']:
	    tb.Refresh(tb['SelectedNode']['Key'])
	
    def RemoveNode(tb=trb2):
	if tb['SelectedNode']:
	    tb.RemoveNode(tb['SelectedNode']['Key'])

    trb2.bind('<<FillLevel>>',FillLevel)
    trb2.bind('<<BeforeIconUpdate>>',BeforeIconUpdate2)
    trb2.bind('<<NodeClick-3>>',NodeClick32)
	
    btn=Button(root,text='Fill',command=trb2.FillTreeBox)
    btn.place(relx=0.5,rely=1,relwidth=0.17,relheight=0.1,anchor=SW)
    btn=Button(root,text='Refresh',command=Refresh)
    btn.place(relx=0.67,rely=1,relwidth=0.16,relheight=0.1,anchor=SW)
    btn=Button(root,text='RemoveNode',command=RemoveNode)
    btn.place(relx=0.83,rely=1,relwidth=0.17,relheight=0.1,anchor=SW)

    root.geometry('450x240')
    root.mainloop()    

if __name__=='__main__':
    test()
