(**************************************************************************)
(*                                                                        *)
(* Module:  Unit 'DelphPyt'            Copyright (c) 1997                 *)
(* Version: 1.0                        IBS Schillings GmbH & Co KG        *)
(* Sub-Version: 0.4                    Ein Unternehmen der KROHNE-Gruppe  *)
(*                                     Heisenbergstr. 18                  *)
(*                                     50169 Kerpen-Trnich               *)
(*                                     Phone: (49)22 37/97 44-0           *)
(*                                                                        *)
(**************************************************************************)
(*  Functionality:  Delphi Python Components                              *)
(*  Changes:  1.2:      error handling in run component                   *)
(*            1.3:      Inserted functionality and definitions got from   *)
(*                      Grzegorz Makarewicz                               *)
(*                      mak@mikroplan.com.pl                              *)
(*                      http://www.mikroplan.com.pl/home/mak/python       *)
(*                      Thank you for permission and help!                *)
(*  	      1.4:	Cleaning up, eliminating German comments, further *)
(*                      tests                                             *)
(*                      Inserted some ideas from Andrew Robinson,         *)
(*                      andy@hps1.demon.co.uk                             *)
(*            1.5:      Initialization String, some minor functionalities *)
(*  Dr. Dietmar Budelsky 1997		                                  *)
(*  budelsky@ibs.bm.eunet.de                                              *)
(*                                                                        *)
(**************************************************************************)
(* This source code is distributed with no WARRANTY, for no reason or use.*)
(* Everyone is allowed to use and change this code free for his own tasks *)
(* and projects, as long as this header and its copyright text is intact. *)
(* For changed versions of this code, which are public distributed the    *)
(* following additional conditions have to be fullfilled:                 *)
(* 1) The header has to contain a comment on the change and the author of *)
(*    it.                                                                 *)
(* 2) A copy of the changed source has to be sent to the above E-Mail     *)
(*    address or my then valid address, if this is possible to the        *)
(*    author.                                                             *)
(* The second condition has the target to maintain an up to date central  *)
(* version of the component. If this condition is not acceptable for      *)
(* confidential or legal reasons, everyone is free to derive a component  *)
(* or to generate a diff file to my or other original sources.            *)
(* Dr. Dietmar Budelsky, 1997-11-17                                       *)
(**************************************************************************)

unit DelphPYT;

interface

uses Classes, Windows;

const
{PYTHON specific constants}
PYTHON_DLL_NAME:PChar = 'python14.dll';
PYTHON_API_VERSION = 1006;
PYT_SCRIPT_BUFFER_INCREASE = 4096;
PYT_METHOD_BUFFER_INCREASE = 10;

CONST
        single_input    = 256;
        file_input      = 257;
        eval_input      = 258;
        funcdef         = 259;
        parameters      = 260;
        varargslist     = 261;
        fpdef           = 262;
        fplist          = 263;
        stmt            = 264;
        simple_stmt     = 265;
        small_stmt      = 266;
        expr_stmt       = 267;
        print_stmt      = 268;
        del_stmt        = 269;
        pass_stmt       = 270;
        flow_stmt       = 271;
        break_stmt      = 272;
        continue_stmt   = 273;
        return_stmt     = 274;
        raise_stmt      = 275;
        import_stmt     = 276;
        dotted_name     = 277;
        global_stmt     = 278;
        exec_stmt       = 279;
        compound_stmt   = 280;
        if_stmt         = 281;
        while_stmt      = 282;
        for_stmt        = 283;
        try_stmt        = 284;
        except_clause   = 285;
        suite           = 286;
        test            = 287;
        and_test        = 288;
        not_test        = 289;
        comparison      = 290;
        comp_op         = 291;
        expr            = 292;
        xor_expr        = 293;
        and_expr        = 294;
        shift_expr      = 295;
        arith_expr      = 296;
        term            = 297;
        factor          = 298;
        power           = 299;
        atom            = 300;
        lambdef         = 301;
        trailer         = 302;
        subscriptlist   = 303;
        subscript       = 304;
        sliceop         = 305;
        exprlist        = 306;
        testlist        = 307;
        dictmaker       = 308;
        classdef        = 309;
        arglist         = 310;
        argument        = 311;

{Non-Python specific constants}
        ErrInit         = -300;

{Global declarations, nothing Python specific and Component for DLL's}
type
   TPChar  = array[0..0] of PChar;
   PPChar  = ^TPChar;
   PInt	   = ^Integer;
   PDouble = ^Double;
   PFloat  = ^Real;
   PLong   = ^LongInt;
   PShort  = ^ShortInt;
   PString = ^PChar;


type TDynDLL = class(TComponent)
  protected
    errorCode  : integer;
    WrongFunc  : PChar;
    DLLHandle  : THandle;
    function TestGetProcAddress(funcname: PChar): Pointer;
  public
    constructor create(AOwner: TComponent); override;
    destructor  destroy;                    override;
    procedure openDll(const dllName:Pchar);
    property getErrorFunc: PChar   read WrongFunc;
    property getErrorCode: integer read errorCode;
  end;

{Python specific interface}
type
  PP_frozen	 = ^P_frozen;
  P_frozen	 = ^_frozen;
  PPyObject	 = ^PyObject;
  PPPyObject	 = ^PPyObject;
  PPPPyObject	 = ^PPPyObject;
  PPyIntObject	 = ^PyIntObject;
  PPyTypeObject  = ^PyTypeObject;
  PPySliceObject = ^PySliceObject;

  AtExitProc   = procedure;
  THREADPROC   = procedure (a:Pointer);

  PyCFunction = function( self, args:PPyObject): PPyObject; cdecl;

  unaryfunc    = function( ob1 : PPyObject): PPyObject; cdecl;
  binaryfunc   = function( ob1,ob2 : PPyObject): PPyObject; cdecl;
  ternaryfunc  = function(ob1,ob2,ob3 : PPyObject): PPyObject; cdecl;
  inquiry      = function( ob1 : PPyObject): integer; cdecl;
  coercion     = function( ob1,ob2 : PPPyObject): integer; cdecl;
  intargfunc   = function(ob1 : PPyObject; i: integer): PPyObject; cdecl;
  intintargfunc = function(ob1 : PPyObject; i1, i2: integer):
                             PPyObject; cdecl;
  intobjargproc = function(ob1 : PPyObject; i: integer; ob2 : PPyObject):
                            integer; cdecl;
  intintobjargproc  = function(ob1: PPyObject; i1, i2: integer;
                                 ob2: PPyObject): integer; cdecl;
  objobjargproc = function(ob1,ob2,ob3 : PPyObject): integer; cdecl;

  pydestructor = procedure(ob: PPyObject); cdecl;
  printfunc    = function(ob: PPyObject; var f: file; i: integer): integer; cdecl;
  getattrfunc  = function(ob1: PPyObject; name: PChar): PPyObject; cdecl;
  setattrfunc  = function(ob1: PPyObject; name: PChar; ob2: PPyObject): integer; cdecl;
  cmpfunc      = function(ob1,ob2: PPyObject): integer; cdecl;
  reprfunc     = function(ob: PPyObject): PPyObject; cdecl;
  hashfunc     = function(ob: PPyObject): LongInt; cdecl;
  getattrofunc = function(ob1,ob2: PPyObject): PPyObject; cdecl;
  setattrofunc = function(ob1,ob2,ob3: PPyObject): integer; cdecl;

  PyNumberMethods = packed record
     nb_add	  : binaryfunc;
     nb_subtract  : binaryfunc;
     nb_multiply  : binaryfunc;
     nb_divide	  : binaryfunc;
     nb_remainder : binaryfunc;
     nb_divmod	  : binaryfunc;
     nb_power	  : ternaryfunc;
     nb_negative  : unaryfunc;
     nb_positive  : unaryfunc;
     nb_absolute  : unaryfunc;
     nb_nonzero	  : inquiry;
     nb_invert	  : unaryfunc;
     nb_lshift	  : binaryfunc;
     nb_rshift	  : binaryfunc;
     nb_and	  : binaryfunc;
     nb_xor	  : binaryfunc;
     nb_or	  : binaryfunc;
     nb_coerce	  : coercion;
     nb_int	  : unaryfunc;
     nb_long	  : unaryfunc;
     nb_float	  : unaryfunc;
     nb_oct	  : unaryfunc;
     nb_hex	  : unaryfunc;
  end;

  PySequenceMethods = packed record
     sq_length	  : inquiry;
     sq_concat	  : binaryfunc;
     sq_repeat	  : intargfunc;
     sq_item	  : intargfunc;
     sq_slice	  : intintargfunc;
     sq_ass_item  : intobjargproc;
     sq_ass_slice : intintobjargproc;
  end;

  PyMappingMethods = packed record
     mp_length	      : inquiry;
     mp_subscript     : binaryfunc;
     mp_ass_subscript : objobjargproc;
  end;

   Py_complex =  packed record
      real : double;
      imag : double;
   end;

PyObject = packed record
  ob_refcnt: Integer;
  ob_type:   PPyTypeObject;
end;

PyIntObject = packed record
  ob_refcnt : Integer;
  ob_type   : PPyTypeObject;
  ob_ival   : LongInt;
end;

_frozen = packed record
   name	: PChar;
   code	: PByte;
   size	: Integer;
end;

PySliceObject = packed record
  ob_refcnt: Integer;
  ob_type:   PPyTypeObject;
  start, stop, step: PPyObject;
end;

PyTypeObject = packed record
  ob_refcnt: Integer;
  ob_type:   PPyTypeObject;
  ob_size:   Integer;
  tp_name:   PChar;
  tp_basicsize, tp_itemsize: Integer;
  tp_dealloc: pydestructor;
  tp_print:   printfunc;
  tp_getattr: getattrfunc;
  tp_setattr: setattrfunc;
  tp_compare: cmpfunc;
  tp_repr:    reprfunc;
  tp_as_number:   PyNumberMethods;
  tp_as_sequence: PySequenceMethods;
  tp_as_mapping:  PyMappingMethods;
  tp_hash:    hashfunc;
  tp_call:    ternaryfunc;
  tp_str:     reprfunc;
  tp_getattro: getattrofunc;
  tp_setattro: setattrofunc;
  tp_xxx3: LongInt;
  tp_xxx4: LongInt;
  tp_doc:  PChar;
end;

PPyMethodDef = ^PyMethodDef;
PyMethodDef  = packed record
   ml_name:  PChar;
   ml_meth:  PyCFunction;
   ml_flags: Integer;
   ml_doc:   PChar;
end;
PPyMethodChain = ^PyMethodChain;
PyMethodChain = packed record
  methods: PPyMethodDef;
  link:    PPyMethodChain;
end;

type TPytDll=class(TDynDll)
private
  DLL_PyArg_Parse: function( args: PPyObject; format: PChar {;....}):
                   Integer; cdecl;
  DLL_PyArg_ParseTuple:
                   function( args: PPyObject; format: PChar {;...}):
                   Integer; cdecl;
  DLL_Py_BuildValue:
                   function( format: PChar {;...}): PPyObject; cdecl;
public
  _PySys_CheckInterval: PInt;
  Py_DebugFlag: PInt;
  Py_VerboseFlag: PInt;
  Py_SuppressPrintingFlag: PInt;

  _PySys_TraceFunc:    PPPyObject;
  _PySys_ProfileFunc: PPPPyObject;

  PyImport_FrozenModules: PP_frozen;


  Py_None:            PPyObject;
  Py_Ellipsis:        PPyObject;
  Py_False:           PPyIntObject;
  Py_True:            PPyIntObject;

  PyExc_AccessError: PPPyObject;
  PyExc_AttributeError: PPPyObject;
  PyExc_ConflictError: PPPyObject;
  PyExc_EOFError: PPPyObject;
  PyExc_IOError: PPPyObject;
  PyExc_ImportError: PPPyObject;
  PyExc_IndexError: PPPyObject;
  PyExc_KeyError: PPPyObject;
  PyExc_KeyboardInterrupt: PPPyObject;
  PyExc_MemoryError: PPPyObject;
  PyExc_NameError: PPPyObject;
  PyExc_OverflowError: PPPyObject;
  PyExc_RuntimeError: PPPyObject;
  PyExc_SyntaxError: PPPyObject;
  PyExc_SystemError: PPPyObject;
  PyExc_SystemExit: PPPyObject;
  PyExc_TypeError: PPPyObject;
  PyExc_ValueError: PPPyObject;
  PyExc_ZeroDivisionError: PPPyObject;

  PyType_Type: PPyTypeObject;
  PyCFunction_Type: PPyTypeObject;
  PyCObject_Type: PPyTypeObject;
  PyClass_Type: PPyTypeObject;
  PyCode_Type: PPyTypeObject;
  PyComplex_Type: PPyTypeObject;
  PyDict_Type: PPyTypeObject;
  PyFile_Type: PPyTypeObject;
  PyFloat_Type: PPyTypeObject;
  PyFrame_Type: PPyTypeObject;
  PyFunction_Type: PPyTypeObject;
  PyInstance_Type: PPyTypeObject;
  PyInt_Type: PPyTypeObject;
  PyList_Type: PPyTypeObject;
  PyLong_Type: PPyTypeObject;
  PyMethod_Type: PPyTypeObject;
  PyModule_Type: PPyTypeObject;
  PyObject_Type: PPyTypeObject;
  PyRange_Type: PPyTypeObject;
  PySlice_Type: PPyTypeObject;
  PyString_Type: PPyTypeObject;
  PyTuple_Type: PPyTypeObject;

  PyArg_GetObject: function(args : PPyObject; nargs, i: integer; p_a: PPPyObject): integer; cdecl;
  PyArg_GetLong:   function(args : PPyObject; nargs, i: integer; p_a: PLong): integer; cdecl;
  PyArg_GetShort:  function(args : PPyObject; nargs, i: integer; p_a: PShort): integer; cdecl;
  PyArg_GetFloat:  function(args : PPyObject; nargs, i: integer; p_a: PFloat): integer; cdecl;
  PyArg_GetString: function(args : PPyObject; nargs, i: integer; p_a: PString): integer; cdecl;
  PyArgs_VaParse:  function (args : PPyObject; format: PChar; va_list: array of const): integer; cdecl;
{Does not work!}
  {Py_VaBuildValue: function (format: PChar; va_list: array of const): PPyObject; cdecl;}
  PyBuiltin_Init:     procedure; cdecl;

  PyComplex_FromCComplex: function(c: Py_complex):PPyObject; cdecl;
  PyComplex_FromDoubles: function(realv,imag : double):PPyObject; cdecl;
  PyComplex_RealAsDouble: function(op : PPyObject ): double; cdecl;
  PyComplex_ImagAsDouble: function(op : PPyObject ): double; cdecl;
  PyComplex_AsCComplex: function(op : PPyObject ): Py_complex; cdecl;
  PyCFunction_GetFunction: function(ob : PPyObject): Pointer; cdecl;
  PyCFunction_GetSelf: function(ob : PPyObject): PPyObject; cdecl;
  PyCallable_Check: function(ob	: PPyObject): integer; cdecl;
  PyCObject_FromVoidPtr: function(cobj, destruct : Pointer): PPyObject; cdecl;
  PyCObject_AsVoidPtr: function(ob : PPyObject): Pointer; cdecl;
  PyClass_New: function (ob1,ob2,ob3 :  PPyObject): PPyObject; cdecl;
  PyClass_IsSubclass: function (ob1, ob2 : PPyObject): integer cdecl;

  Py_InitModule4: function( name: PChar; methods: PPyMethodDef; doc: PChar;
                            passthrough: PPyObject; Api_Version: Integer):
                                PPyObject; cdecl;
  PyErr_Print: procedure; cdecl;
  PyErr_SetNone: procedure(value: PPyObject); cdecl;
  PyErr_SetObject: procedure  (ob1, ob2	: PPyObject); cdecl;
  PyErr_Restore: procedure  (errtype, errvalue, errtraceback: PPyObject); cdecl;
  PyErr_BadArgument: function: integer; cdecl;
  PyErr_NoMemory: function: PPyObject; cdecl;
  PyErr_SetFromErrno: function (ob :  PPyObject):PPyObject; cdecl;
  PyErr_BadInternalCall: procedure; cdecl;
  PyErr_CheckSignals: function: integer; cdecl;
  PyErr_Occurred:     function: PPyObject; cdecl;
  PyErr_Clear:        procedure; cdecl;
  PyErr_Fetch:        procedure( errtype, errvalue, errtraceback: PPPyObject);
                                 cdecl;
  PyErr_SetString:    procedure( ErrorObject: PPyObject; text: PChar); cdecl;
  PyImport_GetModuleDict: function: PPyObject; cdecl;
  PyInt_FromLong:     function( x: LongInt):PPyObject; cdecl;
  Py_Initialize:      procedure; cdecl;
  Py_Exit:            procedure( RetVal: Integer); cdecl;
  PyEval_GetBuiltins: function: PPyObject; cdecl;
  PyDict_GetItem: function(mp, key : PPyObject):PPyObject; cdecl;
  PyDict_SetItem: function(mp, key, item :PPyObject ):integer; cdecl;
  PyDict_DelItem: function(mp, key : PPyObject ):integer; cdecl;
  PyDict_Clear: procedure(mp : PPyObject); cdecl;
  PyDict_Next: function(mp : PPyObject; pos: PInt; key, value: PPPyObject):integer; cdecl;
  PyDict_Keys: function(mp: PPyObject):PPyObject; cdecl;
  PyDict_Values: function(mp: PPyObject):PPyObject; cdecl;
  PyDict_Items: function(mp: PPyObject):PPyObject; cdecl;
  PyDict_Size: function(mp: PPyObject):integer; cdecl;
  PyDict_DelItemString: function(dp : PPyObject;key : PChar ):integer; cdecl;
  PyDict_New: function: PPyObject; cdecl;
  PyDict_GetItemString: function( dp: PPyObject; key: PChar): PPyObject; cdecl;
  PyDict_SetItemString: function( dp: PPyObject; key: PChar; item: PPyObject):
                        Integer; cdecl;
  PyModule_GetDict:     function( module:PPyObject): PPyObject; cdecl;
  PyObject_Str:         function( v: PPyObject): PPyObject; cdecl;
  PyRun_String:         function( str: PChar; start: Integer; globals: PPyObject;
                                  locals: PPyObject): PPyObject; cdecl;
  PyRun_SimpleString:   function( str: PChar): Integer; cdecl;
  PyString_AsString:    function( ob: PPyObject): PChar; cdecl;
  PyString_FromString:  function( str: PChar): PPyObject; cdecl;
  PySys_SetArgv:        procedure( argc: Integer; argv: PPChar); cdecl;

{+ means, Grzegorz has tested his non object version of this function}
{-}        PyCFunction_New: function(md:PPyMethodDef;ob:PPyObject):PPyObject; cdecl;
{+}        PyEval_CallObject: function(ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyEval_CallObjectWithKeywords:function (ob1,ob2,ob3:PPyObject):PPyObject; cdecl;
{-}        PyEval_GetFrame:function :PPyObject; cdecl;
{-}        PyEval_GetGlobals:function :PPyObject; cdecl;
{-}        PyEval_GetLocals:function :PPyObject; cdecl;
{-}        PyEval_GetOwner:function :PPyObject; cdecl;
{-}        PyEval_GetRestricted:function :integer; cdecl;
{-}        PyEval_InitThreads:procedure; cdecl;
{-}        PyEval_RestoreThread:procedure(ob:PPyObject); cdecl;
{-}        PyEval_SaveThread:function :PPyObject; cdecl;
{-}        PyFile_FromString:function (pc1,pc2:PChar):PPyObject; cdecl;
{-}        PyFile_GetLine:function (ob:PPyObject;i:integer):PPyObject; cdecl;
{-}        PyFile_Name:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyFile_SetBufSize:procedure(ob:PPyObject;i:integer); cdecl;
{-}        PyFile_SoftSpace:function (ob:PPyObject;i:integer):integer; cdecl;
{-}        PyFile_WriteObject:function (ob1,ob2:PPyObject;i:integer):integer; cdecl;
{-}        PyFile_WriteString:procedure(s:PChar;ob:PPyObject); cdecl;
{+}        PyFloat_AsDouble:function (ob:PPyObject):DOUBLE; cdecl;
{+}        PyFloat_FromDouble:function (db:double):PPyObject; cdecl;
{-}        PyFunction_GetCode:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyFunction_GetGlobals:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyFunction_New:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyImport_AddModule:function (name:PChar):PPyObject; cdecl;
{-}        PyImport_Cleanup:procedure; cdecl;
{-}        PyImport_GetMagicNumber:function :LONGINT; cdecl;
{+}        PyImport_ImportFrozenModule:function (key:PChar):integer; cdecl;
{+}        PyImport_ImportModule:function (name:PChar):PPyObject; cdecl;
{-}        PyImport_Init:procedure; cdecl;
{-}        PyImport_ReloadModule:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyInstance_New:function (ob1,ob2,ob3:PPyObject):PPyObject; cdecl;
{+}        PyInt_AsLong:function (ob:PPyObject):LONGINT; cdecl;
{-}        PyList_Append:function (ob1,ob2:PPyObject):integer; cdecl;
{-}        PyList_AsTuple:function (ob:PPyObject):PPyObject; cdecl;
{+}        PyList_GetItem:function (ob:PPyObject;i:integer):PPyObject; cdecl;
{-}        PyList_GetSlice:function (ob:PPyObject;i1,i2:integer):PPyObject; cdecl;
{-}        PyList_Insert:function (dp:PPyObject;key:PChar;item:PPyObject):integer; cdecl;
{-}        PyList_New:function (size:integer):PPyObject; cdecl;
{-}        PyList_Reverse:function (ob:PPyObject):integer; cdecl;
{-}        PyList_SetItem:function (dp:PPyObject;key:PChar;item:PPyObject):integer; cdecl;
{-}        PyList_SetSlice:function (ob:PPyObject;i1,i2:integer;ob2:PPyObject):integer; cdecl;
{+}        PyList_Size:function (ob:PPyObject):integer; cdecl;
{-}        PyList_Sort:function (ob:PPyObject):integer; cdecl;
{-}        PyLong_AsDouble:function (ob:PPyObject):DOUBLE; cdecl;
{+}        PyLong_AsLong:function (ob:PPyObject):LONGINT; cdecl;
{+}        PyLong_FromDouble:function (db:double):PPyObject; cdecl;
{+}        PyLong_FromLong:function (l:longint):PPyObject; cdecl;
{-}        PyLong_FromString:function (pc:PChar;var ppc:PChar;i:integer):PPyObject; cdecl;
{-}        PyMapping_Check:function (ob:PPyObject):integer; cdecl;
{-}        PyMapping_GetItemString:function (ob:PPyObject;key:PChar):PPyObject; cdecl;
{-}        PyMapping_HasKey:function (ob,key:PPyObject):integer; cdecl;
{-}        PyMapping_HasKeyString:function (ob:PPyObject;key:PChar):integer; cdecl;
{-}        PyMapping_Length:function (ob:PPyObject):integer; cdecl;
{-}        PyMapping_SetItemString:function (ob:PPyObject; key:PChar; value:PPyObject):integer; cdecl;
{-}        PyMethod_Class:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyMethod_Function:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyMethod_New:function (ob1,ob2,ob3:PPyObject):PPyObject; cdecl;
{-}        PyMethod_Self:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyModule_GetName:function (ob:PPyObject):PChar; cdecl;
{-}        PyModule_New:function (key:PChar):PPyObject; cdecl;
{-}        PyNumber_Absolute:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Add:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_And:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Check:function (ob:PPyObject):integer; cdecl;
{-}        PyNumber_Coerce:function (var ob1,ob2:PPyObject):integer; cdecl;
{-}        PyNumber_Divide:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Divmod:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Float:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Int:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Invert:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Long:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Lshift:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Multiply:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Negative:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Or:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Positive:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Power:function (ob1,ob2,ob3:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Remainder:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Rshift:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Subtract:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyNumber_Xor:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyOS_InitInterrupts:procedure; cdecl;
{-}        PyOS_InterruptOccurred:function :integer; cdecl;
{-}        PyObject_CallObject:function (ob,args:PPyObject):PPyObject; cdecl;
{-}        PyObject_Compare:function (ob1,ob2:PPyObject):integer; cdecl;
{-}        PyObject_GetAttr:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{+}        PyObject_GetAttrString:function (ob:PPyObject;c:PChar):PPyObject; cdecl;
{-}        PyObject_GetItem:function (ob,key:PPyObject):PPyObject; cdecl;
{-}        PyObject_HasAttrString:function (ob:PPyObject;key:PChar):integer; cdecl;
{-}        PyObject_Hash:function (ob:PPyObject):LONGINT; cdecl;
{-}        PyObject_IsTrue:function (ob:PPyObject):integer; cdecl;
{-}        PyObject_Length:function (ob:PPyObject):integer; cdecl;
{-}        PyObject_Repr:function (ob:PPyObject):PPyObject; cdecl;
{-}        PyObject_SetAttr:function (ob1,ob2,ob3:PPyObject):integer; cdecl;
{-}        PyObject_SetAttrString:function (ob:PPyObject;key:Pchar;value:PPyObject):integer; cdecl;
{-}        PyObject_SetItem:function (ob1,ob2,ob3:PPyObject):integer; cdecl;
{-}        PyRange_New:function (l1,l2,l3:longint;i:integer):PPyObject; cdecl;
{-}        PySequence_Check:function (ob:PPyObject):integer; cdecl;
{-}        PySequence_Concat:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PySequence_Count:function (ob1,ob2:PPyObject):integer; cdecl;
{-}        PySequence_GetItem:function (ob:PPyObject;i:integer):PPyObject; cdecl;
{-}        PySequence_GetSlice:function (ob:PPyObject;i1,i2:integer):PPyObject; cdecl;
{-}        PySequence_In:function (ob1,ob2:PPyObject):integer; cdecl;
{-}        PySequence_Index:function (ob1,ob2:PPyObject):integer; cdecl;
{-}        PySequence_Length:function (ob:PPyObject):integer; cdecl;
{-}        PySequence_Repeat:function (ob:PPyObject;count:integer):PPyObject; cdecl;
{-}        PySequence_SetItem:function (ob:PPyObject;i:integer;value:PPyObject):integer; cdecl;
{-}        PySequence_SetSlice:function (ob:PPyObject;i1,i2:integer;value:PPyObject):integer; cdecl;
{-}        PySequence_Tuple:function (ob:PPyObject):PPyObject; cdecl;
{-}        PySlice_GetIndices:function (ob:PPySliceObject;length:integer;var start,stop,step:integer):integer; cdecl;
{-}        PySlice_New:function (start,stop,step:PPyObject):PPyObject; cdecl;
{-}        PyString_Concat:procedure(var ob1:PPyObject;ob2:PPyObject); cdecl;
{-}        PyString_ConcatAndDel:procedure(var ob1:PPyObject;ob2:PPyObject); cdecl;
{-}        PyString_Format:function (ob1,ob2:PPyObject):PPyObject; cdecl;
{-}        PyString_FromStringAndSize:function (s:PChar;i:integer):PPyObject; cdecl;
{-}        PyString_Size:function (ob:PPyObject):integer; cdecl;
{+}        PySys_GetObject:function (s:PChar):PPyObject; cdecl;
{-}        PySys_Init:procedure; cdecl;
{-}        PySys_SetObject:function (s:PChar;ob:PPyObject):integer; cdecl;
{-}        PySys_SetPath:procedure(path:PChar); cdecl;
{-}        PyTraceBack_Fetch:function :PPyObject; cdecl;
{-}        PyTraceBack_Here:function (p:pointer):integer; cdecl;
{-}        PyTraceBack_Print:function (ob1,ob2:PPyObject):integer; cdecl;
{-}        PyTraceBack_Store:function (ob:PPyObject):integer; cdecl;
{+}        PyTuple_GetItem:function (ob:PPyObject;i:integer):PPyObject; cdecl;
{-}        PyTuple_GetSlice:function (ob:PPyObject;i1,i2:integer):PPyObject; cdecl;
{+}        PyTuple_New:function (size:Integer):PPyObject; cdecl;
{+}        PyTuple_SetItem:function (ob:PPyObject;key:integer;value:PPyObject):integer; cdecl;
{+}        PyTuple_Size:function (ob:PPyObject):integer; cdecl;
{-}        Py_AtExit:function (proc: AtExitProc):integer; cdecl;
{-}        Py_Cleanup:procedure; cdecl;
{-}        Py_CompileString:function (s1,s2:PChar;i:integer):PPyObject; cdecl;
{-}        Py_FatalError:procedure(s:PChar); cdecl;
{-}        Py_FindMethod:function (md:PPyMethodDef;ob:PPyObject;key:PChar):PPyObject; cdecl;
{-}        Py_FindMethodInChain:function (mc:PPyMethodChain;ob:PPyObject;key:PChar):PPyObject; cdecl;
{-}        Py_FlushLine:procedure; cdecl;
{-}        _PyObject_New:function (obt:PPyTypeObject;ob:PPyObject):PPyObject; cdecl;
{-}        _PyString_Resize:function (var ob:PPyObject;i:integer):integer; cdecl;
{-}        exit_thread:procedure; cdecl;
{-}        get_thread_ident:function :LONGINT; cdecl;
{-}        init_thread:procedure; cdecl;
{-}        start_new_thread:function (proc:THREADPROC;args:pointer):integer; cdecl;

{Further exported Objects, may be implemented later}
{
    PyCode_New: Pointer;
    PyErr_SetInterrupt: Pointer;
    PyEval_EvalCode: Pointer;
    PyFile_AsFile: Pointer;
    PyFile_FromFile: Pointer;
    PyFloat_AsString: Pointer;
    PyFrame_BlockPop: Pointer;
    PyFrame_BlockSetup: Pointer;
    PyFrame_ExtendStack: Pointer;
    PyFrame_FastToLocals: Pointer;
    PyFrame_LocalsToFast: Pointer;
    PyFrame_New: Pointer;
    PyGrammar_AddAccelerators: Pointer;
    PyGrammar_FindDFA: Pointer;
    PyGrammar_LabelRepr: Pointer;
    PyInstance_DoBinOp: Pointer;
    PyInt_GetMax: Pointer;
    PyMarshal_Init: Pointer;
    PyMarshal_ReadLongFromFile: Pointer;
    PyMarshal_ReadObjectFromFile: Pointer;
    PyMarshal_ReadObjectFromString: Pointer;
    PyMarshal_WriteLongToFile: Pointer;
    PyMarshal_WriteObjectToFile: Pointer;
    PyMember_Get: Pointer;
    PyMember_Set: Pointer;
    PyNode_AddChild: Pointer;
    PyNode_Compile: Pointer;
    PyNode_Free: Pointer;
    PyNode_New: Pointer;
    PyOS_GetLastModificationTime: Pointer;
    PyOS_Readline: Pointer;
    PyOS_strtol: Pointer;
    PyOS_strtoul: Pointer;
    PyObject_CallFunction: Pointer;
    PyObject_CallMethod: Pointer;
    PyObject_Print: Pointer;
    PyParser_AddToken: Pointer;
    PyParser_Delete: Pointer;
    PyParser_New: Pointer;
    PyParser_ParseFile: Pointer;
    PyParser_ParseString: Pointer;
    PyParser_SimpleParseFile: Pointer;
    PyParser_SimpleParseString: Pointer;
    PyRun_AnyFile: Pointer;
    PyRun_File: Pointer;
    PyRun_InteractiveLoop: Pointer;
    PyRun_InteractiveOne: Pointer;
    PyRun_SimpleFile: Pointer;
    PySys_GetFile: Pointer;
    PyToken_OneChar: Pointer;
    PyToken_TwoChars: Pointer;
    PyTokenizer_Free: Pointer;
    PyTokenizer_FromFile: Pointer;
    PyTokenizer_FromString: Pointer;
    PyTokenizer_Get: Pointer;
    Py_GetCopyright: Pointer;
    Py_GetExecPrefix: Pointer;
    Py_GetPath: Pointer;
    Py_GetPrefix: Pointer;
    Py_GetProgramName: Pointer;
    Py_GetVersion: Pointer;
    Py_Main: Pointer;
    _PyObject_NewVar: Pointer;
    _PyParser_Grammar: Pointer;
    _PyParser_TokenNames: Pointer;
    _PyThread_Started: Pointer;
    _Py_c_diff: Pointer;
    _Py_c_neg: Pointer;
    _Py_c_pow: Pointer;
    _Py_c_prod: Pointer;
    _Py_c_quot: Pointer;
    _Py_c_sum: Pointer;
}

  procedure   Py_INCREF   ( op: PPyObject);
  procedure   Py_DECREF   ( op: PPyObject);
  procedure   Py_XINCREF  ( op: PPyObject);
  procedure   Py_XDECREF  ( op: PPyObject);
  function PyArg_Parse     ( args: PPyObject; format: PChar;
                             argp: array of Pointer): Integer; cdecl;
  function PyArg_ParseTuple( args: PPyObject; format: PChar;
                             argp: array of Pointer): Integer; cdecl;
{Danger! do not use with doubles! works only for up to 4 byte types!}
  function Py_BuildValue( format: PChar; args: array of const): PPyObject; cdecl;

  constructor create(AOwner: TComponent); override;
  function    GetPytError: String;
end;

type
PPyDLLObject = ^TPyDLLObject;
TPyDLLObject = packed record
  ob_refcnt: Integer;
  ob_type:   PPyTypeObject;
  PYTDLL:    TPYTDLL;
  ErrorObject: PPyObject;
  m_methods:   PPyMethodDef;
  m_name:      PChar;
end;
// Pytrunobject providing interface for running Python into Delphi
type TPytRun = class(TPytDLL)
private
  fscrbufptr:  PChar;
  fscrbufsize: Integer;
  fscrsize:    Integer;
  fscractptr:  PChar;
  fscrdelim:   String;

  fModuleList:    tlist;
  fMethods:       PPyMethodDef;
  fMethodNum:     integer;
  fMethodBufSize: integer;
  fMethodUsed:    Boolean;

  fInitialized: Boolean;

  FFatalMsgDlg: Boolean;
  FFatalAbort:  Boolean;

  FInitScript:  TStrings;
protected
    procedure SetInitScript(Value: TStrings);
public
  constructor create(AOwner: TComponent); override;
  destructor destroy; override;
  procedure  clearscript;
  procedure  addscript(const source: String);
  property   getscript:PChar read fscrbufptr;
  procedure  clearmethods;
  procedure  addmethod( name: PChar; meth:  PyCFunction;
                        flags: Integer; doc:PChar);
  property   getmethods:PPyMethodDef read fMethods;
  function InitModule( name: PChar; methods: PPyMethodDef; doc: PChar;
                       ProgramName: PChar): PPyObject;
  function IsType(ob: PPyObject; obt: PPyTypeObject): Boolean;
  function GetAttrString(obj: PPyObject; name: PChar):PChar;
published
  property InitScript: TStrings read FInitScript write SetInitScript stored True;
  property FatalMsgDlg: Boolean read FFatalMsgDlg write FFatalMsgDlg
           stored True default True;
  property FatalAbort:  Boolean read FFatalAbort write FFatalAbort
           stored True default True;
end;


function GetPytDLL(ob: PPyObject): TPytRun;
function GetErrorObject(ob: PPyObject): PPyObject;
procedure Register;

implementation

uses SysUtils, Dialogs;

{TPytRun Objects}
constructor TPytRun.create(AOwner: TComponent);
begin
  inherited create(AOwner);
  FatalMsgDlg := True;
  FatalAbort  := True;
  fInitialized := False;
  if (ErrorCode <> NO_ERROR) then begin
    if FFatalMsgDlg then begin
      MessageDlg( 'Error '+IntToStr(ErrorCode)+'; '+getErrorFunc+
                  ': Can not initialize Python! ',mtError,[mbOk],0);
    end;
    if FFatalAbort then halt(1);
    exit;
  end;
  FInitScript := TstringList.Create;
  Py_Initialize;
  fscrbufptr := nil;
  self.clearscript;
  fMethods   := nil;
  self.clearmethods;
  fModuleList := tlist.Create;
  fInitialized := True;
end;

destructor TPytRun.destroy;
var
  i: integer;
  PyDLLObject:PPyDLLObject;
begin
  if fInitialized then begin
    self.clearscript;
    if not(FMethodUsed) then FreeMem(fMethods);
    for i:=0 to fModuleList.Count-1 do begin
        PyDLLObject:=PPyDLLObject(fModuleList.Items[i]);
      with PyDLLObject^ do begin
        FreeMem(m_methods);
        Py_XDECREF(ErrorObject);
      end;
      dispose(PyDLLObject);
    end;
    fModuleList.clear;
    fModuleList.destroy;
    FInitScript.Destroy;
  end;
  inherited destroy;
end;

procedure TPytRun.SetInitScript(Value: TStrings);
begin
  FInitScript.Assign(Value);
end;

function TPytRun.IsType(ob: PPyObject; obt: PPyTypeObject): Boolean;
begin
  result := ob^.ob_type = obt;
end;

function TPytRun.GetAttrString(obj: PPyObject; name: PChar):PChar;
var
   attr: PPyObject;
begin
  result := nil;
  attr := PyObject_GetAttrString(obj,name);
  if attr <> nil then begin
    result := PyString_AsString(attr);
    Py_XDECREF(attr);
  end;
  PyErr_Clear;
end;

procedure  TPytRun.clearscript;
begin
  fscrdelim := '';
  fscrbufsize := 0;
  fscrsize := 0;
  reallocmem(fscrbufptr,fscrbufsize);
  fscractptr := fscrbufptr;
end;


procedure  TPytRun.addscript(const source: String);
var len: Integer;
begin
  len := length(fscrdelim)+length(source);
  if fscrsize+len+1 > fscrbufsize then begin
    fscrbufsize := fscrbufsize + len + PYT_SCRIPT_BUFFER_INCREASE;
    reallocmem(fscrbufptr,fscrbufsize);
    fscrbufptr[fscrsize] := Char(0);
    fscractptr := StrEnd(fscrbufptr);
  end;
  fscractptr := StrEnd(StrPCopy(fscractptr,fscrdelim+source));
  fscrsize := fscrsize + len;
  fscrdelim := #10;
end;

procedure  TPytRun.clearmethods;
begin
  fMethodUsed := False;
  fMethods := PPyMethodDef(AllocMem(SizeOf(PyMethodDef)));
{ AllocMem initializes Memory to zero}
{  with fMethods^ do begin
    ml_name  := nil;
    ml_meth  := nil;
    ml_flags := 0;
    ml_doc   := nil;
  end;}
  fMethodNum := 0;
  fMethodBufSize := 0;
end;

procedure  TPytRun.addmethod( name: PChar; meth:  PyCFunction;
                        flags: Integer; doc:PChar);
var
  TempMethods: PPyMethodDef;
begin
  if fMethodNum >= fMethodBufSize then begin
    fMethodBufSize := fMethodBufSize + PYT_METHOD_BUFFER_INCREASE;
    ReAllocMem(fMethods,SizeOf(PyMethodDef)*(fMethodBufSize+1));
  end;
  TempMethods := fMethods;
  Inc(TempMethods,fMethodNum);
  with TempMethods^ do begin
    ml_name  := name;
    ml_meth  := meth;
    ml_flags := flags;
    ml_doc   := doc;
  end;
  Inc(TempMethods);
  Inc(fMethodNum);
  with TempMethods^ do begin
    ml_name  := nil;
    ml_meth  := nil;
    ml_flags := 0;
    ml_doc   := nil;
  end;
end;

function TPytRun.InitModule( name: PChar; methods: PPyMethodDef; doc: PChar;
                             ProgramName: PChar): PPyObject;
var
    PytMod:        PPyObject;
    pdict,pval:    PPyObject;
    PyDLLObject:   PPyDLLObject;
    ErrorName: string;
    argv:      array[0..1] of PChar;
    NewCount,i:  integer;
  procedure ErrorTest;
  begin
    if PyErr_Occurred <> nil then begin
      if fFatalMsgDlg then
         MessageDlg( 'Error: Can not initialize Module: '+GetPytError,
                     mtError,[mbOk],0);
      if fFatalAbort then halt(1);
      ErrorCode := ErrInit;
      PyErr_Clear;
    end;
  end;

begin
  new(PyDLLObject);
  if (methods = fMethods) then fMethodUsed := True;
  PytMod := Py_InitModule4( name, methods,
                            doc, PPyObject(PyDllObject), PYTHON_API_VERSION);
  pdict := PyModule_GetDict( PytMod);

  with PyDLLObject^ do begin
    ob_refcnt := 1;
    ob_type := nil;
    PytDll := self;
    m_name := name;
    m_methods := methods;
    ErrorName :=m_name+'.error';
    ErrorObject := PyString_FromString(PChar(ErrorName));
    PyDict_SetItemString( pdict, 'error', ErrorObject);
  end;
  ErrorTest;
  fModuleList.Add(PyDLLObject);
  if ProgramName <> nil then begin
    PyDict_SetItemString(pdict,'__builtins__',PyEval_GetBuiltins);
    argv[0] := ProgramName;
    argv[1] := nil;
    PySys_SetArgv(1,@(argv[0]));

    NewCount := InitScript.Count;
    clearscript;
    for i := 0 to NewCount - 1 do
       addscript(InitScript.Strings[i]);
    if NewCount <> 0 then begin
      pval := PyRun_String(getscript, file_input, pdict, pdict);
      if pval = nil then
        if fFatalMsgDlg then
         MessageDlg(GetPytError,mtError,[mbOk],0);
      Py_XDECREF(pval);
      clearscript;
    end;

    ErrorTest;
  end;
  // Do this for Importing Module manually instead
  // pdict := PyDict_New;
  // PyDict_SetItemString( pdict,'Newton',  PyDict_GetItemString(
  //                           PyImport_GetModuleDict, 'Newton'));
  InitModule := pdict;
end;

{ TPytDll Object Methods}
constructor TPYTDLL.create(AOwner: TComponent);
begin
  inherited create(AOwner);
  openDll(PYTHON_DLL_NAME);
  if errorCode = NO_ERROR then begin
     _PySys_CheckInterval := TestGetProcAddress('_PySys_CheckInterval');
     Py_DebugFlag := TestGetProcAddress('Py_DebugFlag');
     Py_VerboseFlag := TestGetProcAddress('Py_VerboseFlag');
     Py_SuppressPrintingFlag := TestGetProcAddress('Py_SuppressPrintingFlag');
     _PySys_TraceFunc := TestGetProcAddress('_PySys_TraceFunc');
     _PySys_ProfileFunc := TestGetProcAddress('_PySys_ProfileFunc');

     Py_None := TestGetProcAddress( '_Py_NoneStruct');
     Py_Ellipsis := TestGetProcAddress( '_Py_EllipsisObject');
     Py_False := TestGetProcAddress( '_Py_ZeroStruct');
     Py_True := TestGetProcAddress( '_Py_TrueStruct');

    PyImport_FrozenModules := TestGetProcAddress( 'PyImport_FrozenModules');

    PyExc_AccessError := TestGetProcAddress('PyExc_AccessError');
    PyExc_AttributeError := TestGetProcAddress('PyExc_AttributeError');
    PyExc_ConflictError := TestGetProcAddress('PyExc_ConflictError');
    PyExc_EOFError := TestGetProcAddress('PyExc_EOFError');
    PyExc_IOError := TestGetProcAddress('PyExc_IOError');
    PyExc_ImportError := TestGetProcAddress('PyExc_ImportError');
    PyExc_IndexError := TestGetProcAddress('PyExc_IndexError');
    PyExc_KeyError := TestGetProcAddress('PyExc_KeyError');
    PyExc_KeyboardInterrupt := TestGetProcAddress('PyExc_KeyboardInterrupt');
    PyExc_MemoryError := TestGetProcAddress('PyExc_MemoryError');
    PyExc_NameError := TestGetProcAddress('PyExc_NameError');
    PyExc_OverflowError := TestGetProcAddress('PyExc_OverflowError');
    PyExc_RuntimeError := TestGetProcAddress('PyExc_RuntimeError');
    PyExc_SyntaxError := TestGetProcAddress('PyExc_SyntaxError');
    PyExc_SystemError := TestGetProcAddress('PyExc_SystemError');
    PyExc_SystemExit := TestGetProcAddress('PyExc_SystemExit');
    PyExc_TypeError := TestGetProcAddress('PyExc_TypeError');
    PyExc_ValueError := TestGetProcAddress('PyExc_ValueError');
    PyExc_ZeroDivisionError := TestGetProcAddress('PyExc_ZeroDivisionError');

    PyType_Type := TestGetProcAddress('PyType_Type');
    PyCFunction_Type := TestGetProcAddress('PyCFunction_Type');
    PyCObject_Type := TestGetProcAddress('PyCObject_Type');
    PyClass_Type := TestGetProcAddress('PyClass_Type');
    PyCode_Type := TestGetProcAddress('PyCode_Type');
    PyComplex_Type := TestGetProcAddress('PyComplex_Type');
    PyDict_Type := TestGetProcAddress('PyDict_Type');
    PyFile_Type := TestGetProcAddress('PyFile_Type');
    PyFloat_Type := TestGetProcAddress('PyFloat_Type');
    PyFrame_Type := TestGetProcAddress('PyFrame_Type');
    PyFunction_Type := TestGetProcAddress('PyFunction_Type');
    PyInstance_Type := TestGetProcAddress('PyInstance_Type');
    PyInt_Type := TestGetProcAddress('PyInt_Type');
    PyList_Type := TestGetProcAddress('PyList_Type');
    PyLong_Type := TestGetProcAddress('PyLong_Type');
    PyMethod_Type := TestGetProcAddress('PyMethod_Type');
    PyModule_Type := TestGetProcAddress('PyModule_Type');
    PyObject_Type := TestGetProcAddress('PyObject_Type');
    PyRange_Type := TestGetProcAddress('PyRange_Type');
    PySlice_Type := TestGetProcAddress('PySlice_Type');
    PyString_Type := TestGetProcAddress('PyString_Type');
    PyTuple_Type := TestGetProcAddress('PyTuple_Type');

    @PyArg_GetObject := TestGetProcAddress( 'PyArg_GetObject');
    @PyArg_GetLong := TestGetProcAddress( 'PyArg_GetLong');
    @PyArg_GetShort := TestGetProcAddress( 'PyArg_GetShort');
    @PyArg_GetFloat := TestGetProcAddress( 'PyArg_GetFloat');
    @PyArg_GetString := TestGetProcAddress( 'PyArg_GetString');
    @PyArgs_VaParse := TestGetProcAddress( 'PyArgs_VaParse');
    {@Py_VaBuildValue := TestGetProcAddress( 'Py_VaBuildValue');}
    @PyBuiltin_Init := TestGetProcAddress( 'PyBuiltin_Init');
     @PyComplex_FromCComplex := TestGetProcAddress( 'PyComplex_FromCComplex');
     @PyComplex_FromDoubles := TestGetProcAddress('PyComplex_FromDoubles');
     @PyComplex_RealAsDouble := TestGetProcAddress('PyComplex_RealAsDouble');
     @PyComplex_ImagAsDouble := TestGetProcAddress('PyComplex_ImagAsDouble');
     @PyComplex_AsCComplex := TestGetProcAddress('PyComplex_AsCComplex');
     @PyCFunction_GetFunction := TestGetProcAddress('PyCFunction_GetFunction');
     @PyCFunction_GetSelf := TestGetProcAddress('PyCFunction_GetSelf');
     @PyCallable_Check := TestGetProcAddress('PyCallable_Check');
     @PyCObject_FromVoidPtr := TestGetProcAddress('PyCObject_FromVoidPtr');
     @PyCObject_AsVoidPtr := TestGetProcAddress('PyCObject_AsVoidPtr');
     @PyClass_New := TestGetProcAddress('PyClass_New');
     @PyClass_IsSubclass := TestGetProcAddress('PyClass_IsSubclass');
     @PyDict_GetItem := TestGetProcAddress('PyDict_GetItem');
     @PyDict_SetItem := TestGetProcAddress('PyDict_SetItem');
     @PyDict_DelItem := TestGetProcAddress('PyDict_DelItem');
     @PyDict_Clear := TestGetProcAddress('PyDict_Clear');
     @PyDict_Next := TestGetProcAddress('PyDict_Next');
     @PyDict_Keys := TestGetProcAddress('PyDict_Keys');
     @PyDict_Values := TestGetProcAddress('PyDict_Values');
     @PyDict_Items := TestGetProcAddress('PyDict_Items');
     @PyDict_Size := TestGetProcAddress('PyDict_Size');
     @PyDict_DelItemString := TestGetProcAddress('PyDict_DelItemString');
    @Py_InitModule4 := TestGetProcAddress( 'Py_InitModule4');
     @PyErr_Print := TestGetProcAddress('PyErr_Print');
     @PyErr_SetNone := TestGetProcAddress('PyErr_SetNone');
     @PyErr_SetObject := TestGetProcAddress('PyErr_SetObject');
     @PyErr_Restore := TestGetProcAddress('PyErr_Restore');
     @PyErr_BadArgument := TestGetProcAddress('PyErr_BadArgument');
     @PyErr_NoMemory := TestGetProcAddress('PyErr_NoMemory');
     @PyErr_SetFromErrno := TestGetProcAddress('PyErr_SetFromErrno');
     @PyErr_BadInternalCall := TestGetProcAddress('PyErr_BadInternalCall');
     @PyErr_CheckSignals := TestGetProcAddress('PyErr_CheckSignals');
    @PyErr_Occurred := TestGetProcAddress( 'PyErr_Occurred');
    @PyErr_Clear := TestGetProcAddress( 'PyErr_Clear');
    @PyErr_Fetch := TestGetProcAddress( 'PyErr_Fetch');
    @PyErr_SetString := TestGetProcAddress( 'PyErr_SetString');
    @PyEval_GetBuiltins := TestGetProcAddress( 'PyEval_GetBuiltins');
    @PyImport_GetModuleDict := TestGetProcAddress( 'PyImport_GetModuleDict');
    @PyInt_FromLong := TestGetProcAddress( 'PyInt_FromLong');
    @DLL_PyArg_ParseTuple := TestGetProcAddress( 'PyArg_ParseTuple');
    @DLL_PyArg_Parse := TestGetProcAddress( 'PyArg_Parse');
    @DLL_Py_BuildValue := TestGetProcAddress('Py_BuildValue');
    @Py_Initialize := TestGetProcAddress( 'Py_Initialize');
    @PyDict_New := TestGetProcAddress( 'PyDict_New');
    @PyDict_SetItemString := TestGetProcAddress( 'PyDict_SetItemString');
    @PyModule_GetDict := TestGetProcAddress( 'PyModule_GetDict');
    @PyObject_Str := TestGetProcAddress( 'PyObject_Str');
    @PyRun_String := TestGetProcAddress( 'PyRun_String');
    @PyRun_SimpleString := TestGetProcAddress( 'PyRun_SimpleString');
    @PyDict_GetItemString := TestGetProcAddress( 'PyDict_GetItemString');
    @PyString_AsString := TestGetProcAddress( 'PyString_AsString');
    @PyString_FromString := TestGetProcAddress( 'PyString_FromString');
    @PySys_SetArgv := TestGetProcAddress( 'PySys_SetArgv');
    @Py_Exit := TestGetProcAddress( 'Py_Exit');

    @PyCFunction_New:=TestGetProcAddress('PyCFunction_New');
    @PyEval_CallObject:=TestGetProcAddress('PyEval_CallObject');
    @PyEval_CallObjectWithKeywords:=TestGetProcAddress('PyEval_CallObjectWithKeywords');
    @PyEval_GetFrame:=TestGetProcAddress('PyEval_GetFrame');
    @PyEval_GetGlobals:=TestGetProcAddress('PyEval_GetGlobals');
    @PyEval_GetLocals:=TestGetProcAddress('PyEval_GetLocals');
    @PyEval_GetOwner:=TestGetProcAddress('PyEval_GetOwner');
    @PyEval_GetRestricted:=TestGetProcAddress('PyEval_GetRestricted');
    @PyEval_InitThreads:=TestGetProcAddress('PyEval_InitThreads');
    @PyEval_RestoreThread:=TestGetProcAddress('PyEval_RestoreThread');
    @PyEval_SaveThread:=TestGetProcAddress('PyEval_SaveThread');
    @PyFile_FromString:=TestGetProcAddress('PyFile_FromString');
    @PyFile_GetLine:=TestGetProcAddress('PyFile_GetLine');
    @PyFile_Name:=TestGetProcAddress('PyFile_Name');
    @PyFile_SetBufSize:=TestGetProcAddress('PyFile_SetBufSize');
    @PyFile_SoftSpace:=TestGetProcAddress('PyFile_SoftSpace');
    @PyFile_WriteObject:=TestGetProcAddress('PyFile_WriteObject');
    @PyFile_WriteString:=TestGetProcAddress('PyFile_WriteString');
    @PyFloat_AsDouble:=TestGetProcAddress('PyFloat_AsDouble');
    @PyFloat_FromDouble:=TestGetProcAddress('PyFloat_FromDouble');
    @PyFunction_GetCode:=TestGetProcAddress('PyFunction_GetCode');
    @PyFunction_GetGlobals:=TestGetProcAddress('PyFunction_GetGlobals');
    @PyFunction_New:=TestGetProcAddress('PyFunction_New');
    @PyImport_AddModule:=TestGetProcAddress('PyImport_AddModule');
    @PyImport_Cleanup:=TestGetProcAddress('PyImport_Cleanup');
    @PyImport_GetMagicNumber:=TestGetProcAddress('PyImport_GetMagicNumber');
    @PyImport_ImportFrozenModule:=TestGetProcAddress('PyImport_ImportFrozenModule');
    @PyImport_ImportModule:=TestGetProcAddress('PyImport_ImportModule');
    @PyImport_Init:=TestGetProcAddress('PyImport_Init');
    @PyImport_ReloadModule:=TestGetProcAddress('PyImport_ReloadModule');
    @PyInstance_New:=TestGetProcAddress('PyInstance_New');
    @PyInt_AsLong:=TestGetProcAddress('PyInt_AsLong');
    @PyList_Append:=TestGetProcAddress('PyList_Append');
    @PyList_AsTuple:=TestGetProcAddress('PyList_AsTuple');
    @PyList_GetItem:=TestGetProcAddress('PyList_GetItem');
    @PyList_GetSlice:=TestGetProcAddress('PyList_GetSlice');
    @PyList_Insert:=TestGetProcAddress('PyList_Insert');
    @PyList_New:=TestGetProcAddress('PyList_New');
    @PyList_Reverse:=TestGetProcAddress('PyList_Reverse');
    @PyList_SetItem:=TestGetProcAddress('PyList_SetItem');
    @PyList_SetSlice:=TestGetProcAddress('PyList_SetSlice');
    @PyList_Size:=TestGetProcAddress('PyList_Size');
    @PyList_Sort:=TestGetProcAddress('PyList_Sort');
    @PyLong_AsDouble:=TestGetProcAddress('PyLong_AsDouble');
    @PyLong_AsLong:=TestGetProcAddress('PyLong_AsLong');
    @PyLong_FromDouble:=TestGetProcAddress('PyLong_FromDouble');
    @PyLong_FromLong:=TestGetProcAddress('PyLong_FromLong');
    @PyLong_FromString:=TestGetProcAddress('PyLong_FromString');
    @PyMapping_Check:=TestGetProcAddress('PyMapping_Check');
    @PyMapping_GetItemString:=TestGetProcAddress('PyMapping_GetItemString');
    @PyMapping_HasKey:=TestGetProcAddress('PyMapping_HasKey');
    @PyMapping_HasKeyString:=TestGetProcAddress('PyMapping_HasKeyString');
    @PyMapping_Length:=TestGetProcAddress('PyMapping_Length');
    @PyMapping_SetItemString:=TestGetProcAddress('PyMapping_SetItemString');
    @PyMethod_Class:=TestGetProcAddress('PyMethod_Class');
    @PyMethod_Function:=TestGetProcAddress('PyMethod_Function');
    @PyMethod_New:=TestGetProcAddress('PyMethod_New');
    @PyMethod_Self:=TestGetProcAddress('PyMethod_Self');
    @PyModule_GetName:=TestGetProcAddress('PyModule_GetName');
    @PyModule_New:=TestGetProcAddress('PyModule_New');
    @PyNumber_Absolute:=TestGetProcAddress('PyNumber_Absolute');
    @PyNumber_Add:=TestGetProcAddress('PyNumber_Add');
    @PyNumber_And:=TestGetProcAddress('PyNumber_And');
    @PyNumber_Check:=TestGetProcAddress('PyNumber_Check');
    @PyNumber_Coerce:=TestGetProcAddress('PyNumber_Coerce');
    @PyNumber_Divide:=TestGetProcAddress('PyNumber_Divide');
    @PyNumber_Divmod:=TestGetProcAddress('PyNumber_Divmod');
    @PyNumber_Float:=TestGetProcAddress('PyNumber_Float');
    @PyNumber_Int:=TestGetProcAddress('PyNumber_Int');
    @PyNumber_Invert:=TestGetProcAddress('PyNumber_Invert');
    @PyNumber_Long:=TestGetProcAddress('PyNumber_Long');
    @PyNumber_Lshift:=TestGetProcAddress('PyNumber_Lshift');
    @PyNumber_Multiply:=TestGetProcAddress('PyNumber_Multiply');
    @PyNumber_Negative:=TestGetProcAddress('PyNumber_Negative');
    @PyNumber_Or:=TestGetProcAddress('PyNumber_Or');
    @PyNumber_Positive:=TestGetProcAddress('PyNumber_Positive');
    @PyNumber_Power:=TestGetProcAddress('PyNumber_Power');
    @PyNumber_Remainder:=TestGetProcAddress('PyNumber_Remainder');
    @PyNumber_Rshift:=TestGetProcAddress('PyNumber_Rshift');
    @PyNumber_Subtract:=TestGetProcAddress('PyNumber_Subtract');
    @PyNumber_Xor:=TestGetProcAddress('PyNumber_Xor');
    @PyOS_InitInterrupts:=TestGetProcAddress('PyOS_InitInterrupts');
    @PyOS_InterruptOccurred:=TestGetProcAddress('PyOS_InterruptOccurred');
    @PyObject_CallObject:=TestGetProcAddress('PyObject_CallObject');
    @PyObject_Compare:=TestGetProcAddress('PyObject_Compare');
    @PyObject_GetAttr:=TestGetProcAddress('PyObject_GetAttr');
    @PyObject_GetAttrString:=TestGetProcAddress('PyObject_GetAttrString');
    @PyObject_GetItem:=TestGetProcAddress('PyObject_GetItem');
    @PyObject_HasAttrString:=TestGetProcAddress('PyObject_HasAttrString');
    @PyObject_Hash:=TestGetProcAddress('PyObject_Hash');
    @PyObject_IsTrue:=TestGetProcAddress('PyObject_IsTrue');
    @PyObject_Length:=TestGetProcAddress('PyObject_Length');
    @PyObject_Repr:=TestGetProcAddress('PyObject_Repr');
    @PyObject_SetAttr:=TestGetProcAddress('PyObject_SetAttr');
    @PyObject_SetAttrString:=TestGetProcAddress('PyObject_SetAttrString');
    @PyObject_SetItem:=TestGetProcAddress('PyObject_SetItem');
    @PyRange_New:=TestGetProcAddress('PyRange_New');
    @PySequence_Check:=TestGetProcAddress('PySequence_Check');
    @PySequence_Concat:=TestGetProcAddress('PySequence_Concat');
    @PySequence_Count:=TestGetProcAddress('PySequence_Count');
    @PySequence_GetItem:=TestGetProcAddress('PySequence_GetItem');
    @PySequence_GetSlice:=TestGetProcAddress('PySequence_GetSlice');
    @PySequence_In:=TestGetProcAddress('PySequence_In');
    @PySequence_Index:=TestGetProcAddress('PySequence_Index');
    @PySequence_Length:=TestGetProcAddress('PySequence_Length');
    @PySequence_Repeat:=TestGetProcAddress('PySequence_Repeat');
    @PySequence_SetItem:=TestGetProcAddress('PySequence_SetItem');
    @PySequence_SetSlice:=TestGetProcAddress('PySequence_SetSlice');
    @PySequence_Tuple:=TestGetProcAddress('PySequence_Tuple');
    @PySlice_GetIndices:=TestGetProcAddress('PySlice_GetIndices');
    @PySlice_New:=TestGetProcAddress('PySlice_New');
    @PyString_Concat:=TestGetProcAddress('PyString_Concat');
    @PyString_ConcatAndDel:=TestGetProcAddress('PyString_ConcatAndDel');
    @PyString_Format:=TestGetProcAddress('PyString_Format');
    @PyString_FromStringAndSize:=TestGetProcAddress('PyString_FromStringAndSize');
    @PyString_Size:=TestGetProcAddress('PyString_Size');
    @PySys_GetObject:=TestGetProcAddress('PySys_GetObject');
    @PySys_Init:=TestGetProcAddress('PySys_Init');
    @PySys_SetObject:=TestGetProcAddress('PySys_SetObject');
    @PySys_SetPath:=TestGetProcAddress('PySys_SetPath');
    @PyTraceBack_Fetch:=TestGetProcAddress('PyTraceBack_Fetch');
    @PyTraceBack_Here:=TestGetProcAddress('PyTraceBack_Here');
    @PyTraceBack_Print:=TestGetProcAddress('PyTraceBack_Print');
    @PyTraceBack_Store:=TestGetProcAddress('PyTraceBack_Store');
    @PyTuple_GetItem:=TestGetProcAddress('PyTuple_GetItem');
    @PyTuple_GetSlice:=TestGetProcAddress('PyTuple_GetSlice');
    @PyTuple_New:=TestGetProcAddress('PyTuple_New');
    @PyTuple_SetItem:=TestGetProcAddress('PyTuple_SetItem');
    @PyTuple_Size:=TestGetProcAddress('PyTuple_Size');
    @Py_AtExit:=TestGetProcAddress('Py_AtExit');
    @Py_Cleanup:=TestGetProcAddress('Py_Cleanup');
    @Py_CompileString:=TestGetProcAddress('Py_CompileString');
    @Py_FatalError:=TestGetProcAddress('Py_FatalError');
    @Py_FindMethod:=TestGetProcAddress('Py_FindMethod');
    @Py_FindMethodInChain:=TestGetProcAddress('Py_FindMethodInChain');
    @Py_FlushLine:=TestGetProcAddress('Py_FlushLine');
    @_PyObject_New:=TestGetProcAddress('_PyObject_New');
    @_PyString_Resize:=TestGetProcAddress('_PyString_Resize');
    @exit_thread:=TestGetProcAddress('exit_thread');
    @get_thread_ident:=TestGetProcAddress('get_thread_ident');
    @init_thread:=TestGetProcAddress('init_thread');
    @start_new_thread:=TestGetProcAddress('start_new_thread');
  end;
end;

function TPYTDLL.GetPytError: String;
var
  errtype, errvalue, errtraceback: PPyObject;
  StrObj : PPyObject;
  temp  : String;
  tchar : PChar;
begin
  PyErr_Fetch(@errtype,@errvalue,@errtraceback);
  if errtype = nil then temp := '<NULL>' else begin
    StrObj := PyObject_Str(errtype);
    PyArg_Parse(StrObj,'s',[@tchar]);
    temp := tchar;
    Py_DECREF(StrObj);
  end;
  temp := temp + ': ';
  if errtype = nil then temp := temp+'<NULL>' else begin
    StrObj := PyObject_Str(errvalue);
    PyArg_Parse(StrObj,'s',[@tchar]);
    temp := temp + tchar;
    Py_DECREF(StrObj);
  end;
  GetPytError := temp;
  PyErr_Clear;
end;

procedure TPYTDLL.Py_INCREF(op: PPyObject);
begin
  Inc(op^.ob_refcnt);
end;

procedure TPYTDLL.Py_DECREF(op: PPyObject);
begin
  with op^ do begin
    Dec(ob_refcnt);
    if ob_refcnt = 0 then begin
      ob_type^.tp_dealloc(op);
    end;
  end;
end;

procedure TPYTDLL.Py_XINCREF(op: PPyObject);
begin
  if op <> nil then Py_INCREF(op);
end;

procedure TPYTDLL.Py_XDECREF(op: PPyObject);
begin
  if op <> nil then Py_DECREF(op);
end;

{ function separates 4 Byte values from the args parameter. The high long
  of an 8 byte type (normally only Double floats) is ignored!}
function TPYTDLL.Py_BuildValue( format: PChar; args: array of const):
         PPyObject; assembler; cdecl;
asm
        mov eax,[args]       // gets args pointer
        mov edx,[args+$4]    // gets invisible argument count-1 parameter
@loop1: mov ecx,[eax+edx*8]  // get one argument
        dec edx              // go to the next argument
        push ecx             // push 4 bytes of the argument
        jns  @loop1          // next argument, if any
        mov  eax,self        // get invisible self parameter
        mov  edx,format      // call DLL Py_Builtin function
        push edx
        call [eax+DLL_Py_BuildValue]
        pop  ecx             // Dummy pop of result high 4 bytes
end;

{DELPHI does the right thing here. It automatically generates
 a copy of the argp on the stack}
function TPYTDLL.PyArg_Parse ( args: PPyObject; format: PChar;
                       argp: array of Pointer): Integer; cdecl;
begin
  PyArg_Parse := DLL_PyArg_Parse(args, format);
end;

function TPYTDLL.PyArg_ParseTuple ( args: PPyObject; format: PChar;
                            argp: array of Pointer): Integer; cdecl;
begin
  PyArg_ParseTuple := DLL_PyArg_ParseTuple(args, format);
end;

//----TDynDLL-----------------------------------------------------------------


  procedure  TDynDLL.openDll(const dllName:PChar);
  begin
     DLLHandle := LoadLibrary(dllName);
     if DLLHandle < 32 then errorCode := GetLastError
     else errorCode := NO_ERROR;
  end;

 constructor TDynDLL.create(AOwner: TComponent);
 begin
   inherited create(AOwner);
   errorCode := ErrInit;
   WrongFunc := '';
 end;

  destructor TDynDLL.destroy;
  begin
   if DLLHandle >= 32 then
      FreeLibrary(DLLHandle);
   inherited destroy;
  end;

function TDynDLL.TestGetProcAddress(funcname: PChar): Pointer;
begin
  result := nil;
  if getErrorCode <> NO_ERROR then exit;
  result := GetProcAddress( DLLHandle, funcname);
  if (result = nil) and (errorCode = NO_ERROR) then begin
    errorCode := GetLastError;
    WrongFunc := funcname;
  end;
end;

{Get the calling Python Delphi object}
function GetPytDLL(ob: PPyObject): TPytRun;
begin
  GetPytDLL := TPytRun(PPyDLLObject(ob)^.PytDll);
end;

{Get the module's Error object}
function GetErrorObject(ob: PPyObject): PPyObject;
begin
  GetErrorObject := PPyDLLObject(ob)^.ErrorObject;
end;

procedure Register;
begin
  RegisterComponents('Python',[TPytDll,TPytRun]);
end;

end.
