#include "Python.h"
#define __MSC__
#include <glide.h>

#define MAX_VERTEX_OFFSET (31)
static float fixedArray[(MAX_VERTEX_OFFSET+1)*5];
static float *vertArray = fixedArray;
static long vertSize;

static int packedRGBOffset;

static PyObject *pg_grSstWinOpen(PyObject *self, PyObject *args)
{
	FxU32 hWnd;
	long screenRes;
	long refresh;
	long colorFormat;
	long originLocation;
	long nColBuffers, nAuxBuffers;
	long context;

	if (!PyArg_ParseTuple(args, "lllllll",
		&hWnd, &screenRes,
		&refresh, &colorFormat,
		&originLocation, &nColBuffers,
		&nAuxBuffers))
		return NULL;

	context = grSstWinOpen(hWnd, screenRes,
				refresh, colorFormat,
				originLocation, nColBuffers,
				nAuxBuffers);
	return PyInt_FromLong((long)context);
}

static PyObject *pg_grSstWinClose(PyObject *self, PyObject *args)
{
	GrContext_t context;
	FxBool b;
	if (!PyArg_ParseTuple(args, "l", &context))
		return NULL;
	b = grSstWinClose(context);
	return PyInt_FromLong((long)b);
}

static PyObject *pg_grSstSelect(PyObject *self, PyObject *args)
	
{
	int selection;
	if (!PyArg_ParseTuple(args, "l", &selection))
		return NULL;
	grSstSelect(selection);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grGlideInit(PyObject *self, PyObject *args)
	
{
	grGlideInit();
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grGlideShutdown(PyObject *self, PyObject *args)
	
{
	grGlideShutdown();
	if (vertArray) free(vertArray);
	vertArray = NULL;
	vertSize = 0;
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grBufferClear(PyObject *self, PyObject *args)
	
{
	GrColor_t color;
	GrAlpha_t alpha;
	FxU32 depth;
	if (!PyArg_ParseTuple(args, "lll", &color, &alpha, &depth))
		return NULL;
	grBufferClear(color, alpha, depth);
	Py_INCREF(Py_None);
	return Py_None;
}
 
static PyObject *pg_grBufferSwap(PyObject *self, PyObject *args)
	
{
	FxU32 interval;
	if (!PyArg_ParseTuple(args, "l", &interval))
		return NULL;
	grBufferSwap(interval);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grRenderBuffer(PyObject *self, PyObject *args)
	
{
	GrBuffer_t buffer;
	if (!PyArg_ParseTuple(args, "l", &buffer))
		return NULL;
	grRenderBuffer(buffer);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grFinish(PyObject *self, PyObject *args)
	
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	grFinish();
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grFlush(PyObject *self, PyObject *args)
	
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	grFlush();
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grSelectContext(PyObject *self, PyObject *args)
	
{
	GrContext_t context;
	FxBool answer;
	if (!PyArg_ParseTuple(args, "l", &context))
		return NULL;
	answer = grSelectContext(context);
	return PyInt_FromLong((long)answer);
}

static PyObject *pg_grSstOrigin(PyObject *self, PyObject *args)
	
{
	GrOriginLocation_t origin;
	if (!PyArg_ParseTuple(args, "l", &origin))
		return NULL;
	grSstOrigin(origin);
	Py_INCREF(Py_None);
	return Py_None;
}

static int resizeBuffer(int newSize)
{
	if (vertSize >= newSize)
		return 1;
	if (vertArray != fixedArray)
		free(vertArray);
	vertSize = 5;
	vertArray = malloc(newSize * (MAX_VERTEX_OFFSET+1) * sizeof(float));
	if (!vertArray)
	{
		vertArray = fixedArray;
		return 0;
	}
	vertSize = newSize;
	return 1;
}

/* Converts the items to the vertex buffer, with special case
   for the packed ARGB value.
   Returns the number of items translated, or -1 if failed */
static int listToVertexBuffer(float *vptr, PyObject *pyobj)
{
	int lsize, i;
	float f;
	PyObject *val, *val2;
	lsize = PyList_Size(pyobj);
	if (lsize > MAX_VERTEX_OFFSET+1) lsize = MAX_VERTEX_OFFSET+1;
	for(i=0; i<lsize; i++)
	{
		val = PyList_GET_ITEM(pyobj, i);
		if (i == packedRGBOffset)
		{
			long *intptr = (long *)(vptr + i);
			val2 = PyNumber_Int(val);
			if (!val2)
			{
				PyErr_SetString(PyExc_TypeError, "vertex list");
				return -1;
			}
			*intptr = PyInt_AS_LONG(val2);
			Py_DECREF(val2);
		}
		else
		{
			val2 = PyNumber_Float(val);
			if (!val2)
			{
				PyErr_SetString(PyExc_TypeError, "vertex list");
				return -1;
			}
			f = (float)PyFloat_AS_DOUBLE(val2);
			vptr[i] = f;
			Py_DECREF(val2);
		}
	}
	return lsize;
}

/* Converts the items to the vertex buffer, with special case
   for the packed ARGB value.
   Returns the number of items translated, or -1 if failed */
static int tupleToVertexBuffer(float *vptr, PyObject *pyobj)
{
	int lsize, i;
	PyObject *val, *val2;
	float f;
	lsize = PyTuple_Size(pyobj);
	if (lsize > MAX_VERTEX_OFFSET+1) lsize = MAX_VERTEX_OFFSET+1;
	for(i=0; i<lsize; i++)
	{
		val = PyTuple_GET_ITEM(pyobj, i);
		if (!val) return -1;
		if (i == packedRGBOffset)
		{
			long *intptr = (long *)(vptr + i);
			val2 = PyNumber_Int(val);
			if (!val2)
			{
				PyErr_SetString(PyExc_TypeError, "vertex list");
				return -1;
			}
			*intptr = PyInt_AS_LONG(val2);
			Py_DECREF(val2);
		}
		else
		{
			val2 = PyNumber_Float(val);
			if (!val)
			{
				PyErr_SetString(PyExc_TypeError, "vertex list");
				return -1;
			}
			f = (float)PyFloat_AS_DOUBLE(val2);
			vptr[i] = f;
			Py_DECREF(val2);
		}
	}
	return lsize;
}

static PyObject *pg_grDrawPoint(PyObject *self, PyObject *args)
	
{
	PyObject *list;
	if (!PyArg_ParseTuple(args, "O", &list))
		return NULL;
	if (!PyList_Check(list))
	{
		if (!PyTuple_Check(list))
			return NULL;
		if (tupleToVertexBuffer(vertArray, list) == -1)
			return NULL;
	}
	else
		if (listToVertexBuffer(vertArray, list) == -1)
			return NULL;
	grDrawPoint(vertArray);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDrawLine(PyObject *self, PyObject *args)
	
{
	PyObject *vert1, *vert2;
	if (!PyArg_ParseTuple(args, "OO", &vert1, &vert2))
		return NULL;
	if (PyList_Check(vert1))
		if (listToVertexBuffer(vertArray, vert1) == -1)
			return NULL;
	else if (PyTuple_Check(vert1))
		if (tupleToVertexBuffer(vertArray, vert1) == -1)
			return NULL;
	else
	{
		PyErr_SetString(PyExc_TypeError, "expecting list/tuple");
		return NULL;
	}
	if (PyList_Check(vert2))
		if (listToVertexBuffer(vertArray+(MAX_VERTEX_OFFSET+1), vert2) == -1)
			return NULL;
	else if (PyTuple_Check(vert2))
		if (tupleToVertexBuffer(vertArray+(MAX_VERTEX_OFFSET+1), vert2) == -1)
			return NULL;
	else
	{
		PyErr_SetString(PyExc_TypeError, "expecting list/tuple");
		return NULL;
	}
	grDrawLine(vertArray, vertArray+(MAX_VERTEX_OFFSET+1));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDrawTriangle(PyObject *self, PyObject *args)
	
{
	PyObject *vert1, *vert2, *vert3;
	if (!PyArg_ParseTuple(args, "OOO", &vert1, &vert2, &vert3))
		return NULL;
	if (PyList_Check(vert1))
		if (listToVertexBuffer(vertArray, vert1) == -1)
			return NULL;
	else if (PyTuple_Check(vert1))
		if (tupleToVertexBuffer(vertArray, vert1) == -1)
			return NULL;
	else
	{
		PyErr_SetString(PyExc_TypeError, "expecting list/tuple");
		return NULL;
	}
	if (PyList_Check(vert2))
		if (listToVertexBuffer(vertArray+(MAX_VERTEX_OFFSET+1), 
				vert2) == -1)
			return NULL;
	else if (PyTuple_Check(vert2))
		if (tupleToVertexBuffer(vertArray+(MAX_VERTEX_OFFSET+1),
				vert2) == -1)
			return NULL;
	else
	{
		PyErr_SetString(PyExc_TypeError, "expecting list/tuple");
		return NULL;
	}
	if (PyList_Check(vert3))
		if (listToVertexBuffer(vertArray+((MAX_VERTEX_OFFSET+1)*2),
				vert3) == -1)
			return NULL;
	else if (PyTuple_Check(vert3))
		if (tupleToVertexBuffer(vertArray+((MAX_VERTEX_OFFSET+1)*2),
				vert3) == -1)
			return NULL;
	else
	{
		PyErr_SetString(PyExc_TypeError, "expecting list/tuple");
		return NULL;
	}
	grDrawTriangle(vertArray, vertArray+(MAX_VERTEX_OFFSET+1),
		vertArray+((MAX_VERTEX_OFFSET+1)*2));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grAADrawTriangle(PyObject *self, PyObject *args)
{
	PyObject *vert1, *vert2, *vert3;
	long ab_aa, bc_aa, ca_aa;
	if (!PyArg_ParseTuple(args, "OOOlll", 
			&vert1, &vert2, &vert3,
			&ab_aa, &bc_aa, ca_aa))
		return NULL;
	if (PyList_Check(vert1))
		if (listToVertexBuffer(vertArray, vert1) == -1)
			return NULL;
	else if (PyTuple_Check(vert1))
		if (tupleToVertexBuffer(vertArray, vert1) == -1)
			return NULL;
	else
	{
		PyErr_SetString(PyExc_TypeError, "expecting list/tuple");
		return NULL;
	}
	if (PyList_Check(vert2))
		if (listToVertexBuffer(vertArray+(MAX_VERTEX_OFFSET+1), 
				vert2) == -1)
			return NULL;
	else if (PyTuple_Check(vert2))
		if (tupleToVertexBuffer(vertArray+(MAX_VERTEX_OFFSET+1),
				vert2) == -1)
			return NULL;
	else
	{
		PyErr_SetString(PyExc_TypeError, "expecting list/tuple");
		return NULL;
	}
	if (PyList_Check(vert3))
		if (listToVertexBuffer(vertArray+((MAX_VERTEX_OFFSET+1)*2),
				vert3) == -1)
			return NULL;
	else if (PyTuple_Check(vert3))
		if (tupleToVertexBuffer(vertArray+((MAX_VERTEX_OFFSET+1)*2),
				vert3) == -1)
			return NULL;
	else
	{
		PyErr_SetString(PyExc_TypeError, "expecting list/tuple");
		return NULL;
	}
	grAADrawTriangle(vertArray, vertArray+(MAX_VERTEX_OFFSET+1),
		vertArray+((MAX_VERTEX_OFFSET+1)*2),
		(FxBool)ab_aa, (FxBool)bc_aa, (FxBool)ca_aa);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDrawVertexArray(PyObject *self, PyObject *args)
	
{
	PyObject *sequence, *seqItem;
	long int mode;
	int length, i;
	if (!PyArg_ParseTuple(args, "lO", &mode, &sequence))
		return NULL;
	if (PyList_Check(sequence))
	{
		length = PyList_Size(sequence);
		if (!resizeBuffer(length))
		{
			PyErr_SetString(PyExc_MemoryError, "Vertex buffer resize");
			return NULL;
		}
		for(i=0; i<length; i++)
		{
			seqItem = PyList_GET_ITEM(sequence, i);
			if (PyList_Check(seqItem))
			{
				if (listToVertexBuffer(vertArray+(i*(MAX_VERTEX_OFFSET+1)),
						seqItem) == -1)
					return NULL;
			else if (PyTuple_Check(seqItem))
				if (tupleToVertexBuffer(vertArray+(i*(MAX_VERTEX_OFFSET+1)),
						seqItem) == -1)
					return NULL;
			}
		}
	}
	else if (PyTuple_Check(sequence))
	{
		length = PyTuple_Size(sequence);
		if (!resizeBuffer(length))
		{
			PyErr_SetString(PyExc_MemoryError, "Vertex buffer resize");
			return NULL;
		}
		for(i=0; i<length; i++)
		{
			seqItem = PyTuple_GET_ITEM(sequence, i);
			if (PyList_Check(seqItem))
			{
				if (listToVertexBuffer(vertArray+(i*(MAX_VERTEX_OFFSET+1)),
						seqItem) == -1)
					return NULL;
			else if (PyTuple_Check(seqItem))
				if (tupleToVertexBuffer(vertArray+(i*(MAX_VERTEX_OFFSET+1)),
						seqItem) == -1)
					return NULL;
			}
		}
	}
	grDrawVertexArrayContiguous(mode, length, vertArray,
		sizeof(float) * (MAX_VERTEX_OFFSET+1));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grVertexLayout(PyObject *self, PyObject *args)

{
	long param, index, mode;
	if (!PyArg_ParseTuple(args, "lll", &param, &index, &mode))
		return NULL;
	if (param == GR_PARAM_PARGB)
	{
		if (mode == GR_PARAM_DISABLE)
			packedRGBOffset = -1;
		else
			packedRGBOffset = index;
	}
	grVertexLayout(param, index*sizeof(float), mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grAlphaBlendFunction(PyObject *self, PyObject *args)

{
	GrAlphaBlendFnc_t rgb_sf, rgb_df, alpha_sf, alpha_df;
	if (!PyArg_ParseTuple(args, "llll", &rgb_sf, &rgb_df, &alpha_sf, &alpha_df))
		return NULL;
	grAlphaBlendFunction(rgb_sf, rgb_df, alpha_sf, alpha_df);
	Py_INCREF(Py_None);
	return Py_None;
}
static PyObject *pg_grAlphaCombine(PyObject *self, PyObject *args)

{
	GrCombineFunction_t function;
	GrCombineFactor_t factor;
	GrCombineLocal_t local;
	GrCombineOther_t other;
	FxBool invert;
	if (!PyArg_ParseTuple(args, "lllll", &function, &factor, &local, &other, &invert))
		return NULL;
	grAlphaCombine(function, factor, local, other, invert);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grAlphaControlsITRGBLighting(PyObject *self, PyObject *args)

{
	FxBool enable;
	if (!PyArg_ParseTuple(args, "l", &enable))
		return NULL;
	grAlphaControlsITRGBLighting(enable);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grAlphaTestFunction(PyObject *self, PyObject *args)

{
	GrCmpFnc_t function;
	if (!PyArg_ParseTuple(args, "l", &function))
		return NULL;
	grAlphaTestFunction(function);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grAlphaTestReferenceValue(PyObject *self, PyObject *args)

{
	long value;
	if (!PyArg_ParseTuple(args, "l", &value))
		return NULL;
	grAlphaTestReferenceValue((GrAlpha_t)value);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grChromakeyMode(PyObject *self, PyObject *args)

{
	GrChromakeyMode_t mode;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	grChromakeyMode(mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grChromakeyValue(PyObject *self, PyObject *args)

{
	GrColor_t value;
	if (!PyArg_ParseTuple(args, "l", &value))
		return NULL;
	grChromakeyValue(value);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grClipWindow(PyObject *self, PyObject *args)

{
	FxU32 minx, miny, maxx, maxy;
	if (!PyArg_ParseTuple(args, "llll", &minx, &miny, &maxx, &maxy))
		return NULL;
	grClipWindow(minx, miny, maxx, maxy);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grColorCombine(PyObject *self, PyObject *args)

{
	long fn, fact, local, other, invert;
	if (!PyArg_ParseTuple(args, "lllll", &fn, &fact, &local, &other, &invert))
		return NULL;
	grColorCombine((GrCombineFunction_t)fn, (GrCombineFactor_t)fact,
		(GrCombineLocal_t)local, (GrCombineOther_t)other, 
		(FxBool)invert);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grColorMask(PyObject *self, PyObject *args)

{
	long rgb, a;
	if (!PyArg_ParseTuple(args, "ll", &rgb, &a))
		return NULL;
	grColorMask((FxBool)rgb, (FxBool)a);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grCullMode(PyObject *self, PyObject *args)

{
	GrCullMode_t mode;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	grCullMode(mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grConstantColorValue(PyObject *self, PyObject *args)

{
	long val;
	if (!PyArg_ParseTuple(args, "l", &val))
		return NULL;
	grConstantColorValue((GrColor_t)val);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDepthBiasLevel(PyObject *self, PyObject *args)

{
	FxI32 level;
	if (!PyArg_ParseTuple(args, "l", &level))
		return NULL;
	grDepthBiasLevel(level);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDepthBufferFunction(PyObject *self, PyObject *args)

{
	long func;
	if (!PyArg_ParseTuple(args, "l", &func))
		return NULL;
	grDepthBufferFunction((GrCmpFnc_t)func);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDepthBufferMode(PyObject *self, PyObject *args)

{
	long mode;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	grDepthBufferMode((GrDepthBufferMode_t)mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDepthMask(PyObject *self, PyObject *args)

{
	long mask;
	if (!PyArg_ParseTuple(args, "l", &mask))
		return NULL;
	grDepthMask((FxBool)mask);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDisableAllEffects(PyObject *self, PyObject *args)

{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	grDisableAllEffects();
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDitherMode(PyObject *self, PyObject *args)

{
	long mode;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	grDitherMode((GrDitherMode_t)mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grFogColorValue(PyObject *self, PyObject *args)

{
	long color;
	if (!PyArg_ParseTuple(args, "l", &color))
		return NULL;
	grFogColorValue((GrColor_t)color);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grFogMode(PyObject *self, PyObject *args)

{
	long mode;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	grFogMode((GrFogMode_t)mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static int getLongFromSequence(long *dest, int i, PyObject *seq)
{
	PyObject *item, *intitem;
	item = PySequence_GetItem(seq, i);
	if (item == NULL)
		return 0;
	intitem = PyNumber_Int(item);
	Py_DECREF(item);
	if (intitem == NULL)
		return 0;
	*dest = PyInt_AS_LONG(intitem);
	Py_DECREF(intitem);
	return 1;
}

static PyObject *pg_grFogTable(PyObject *self, PyObject *args)

{
	PyObject *obj;
	int i;
	FxI32 entries;
	GrFog_t *table;
	grGet(GR_FOG_TABLE_ENTRIES, 4, &entries);
	table = malloc(sizeof(GrFog_t)*entries);
	if (!table)
	{
		PyErr_SetString(PyExc_MemoryError, "fogtable");
		return NULL;
	}
	if (!PyArg_ParseTuple(args, "O", &obj))
	{
		free(table);
		return NULL;
	}
	for(i=0; i<entries; i++)
	{
		long l;
		if (!getLongFromSequence(&l, i, obj))
			{free(table); return NULL;}
		table[i] = (GrFog_t)l;
	}
	grFogTable(table);
	free(table);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grLoadGammaTable(PyObject *self, PyObject *args)

{
	PyObject *redl, *greenl, *bluel;
	long entries;
	FxU32 *table;
	int i;
	if (!PyArg_ParseTuple(args, "lOOO", &entries,
			&redl, &greenl, &bluel))
		return NULL;
	table = malloc(sizeof(FxU32)*entries*3);
	if (!table) {
		PyErr_SetString(PyExc_MemoryError, "Gamma table");
		return NULL;
	}
	for(i=0; i<entries; i++)
	{
		long l1, l2, l3;
		if ((!getLongFromSequence(&l1, i, redl))||
			(!getLongFromSequence(&l2, i, greenl))||
			(!getLongFromSequence(&l3, i, bluel)))
		{
			free(table); return NULL;
		}
		table[i] = l1;
		table[i + entries] = l2;
		table[i + (entries*2)] = l3;
	}
	grLoadGammaTable(entries, table, table+entries,
					 table+(entries*2));
	free(table);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grSplash(PyObject *self, PyObject *args)

{
	double x, y, width, height;
	FxU32 frame;
	if (!PyArg_ParseTuple(args, "ddddl", 
		&x, &y, &width, &height, &frame))
		return NULL;
	grSplash((float)x, (float)y, (float)width,
			 (float)height, frame);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grGet(PyObject *self, PyObject *args)

{
	long pname;
	FxI32 quickArray[4];
	if (!PyArg_ParseTuple(args, "l", &pname))
		return NULL;
	switch (pname)
	{
	case GR_BITS_DEPTH:
	case GR_BITS_GAMMA:
	case GR_FOG_TABLE_ENTRIES:
	case GR_GAMMA_TABLE_ENTRIES:
	case GR_GLIDE_STATE_SIZE:
	case GR_GLIDE_VERTEXLAYOUT_SIZE:
	case GR_IS_BUSY:
	case GR_LFB_PIXEL_PIPE:
	case GR_MAX_TEXTURE_SIZE:
	case GR_MAX_TEXTURE_ASPECT_RATIO:
	case GR_MEMORY_FB:
	case GR_MEMORY_TMU:
	case GR_MEMORY_UMA:
	case GR_NON_POWER_OF_TWO_TEXTURES:
	case GR_NUM_BOARDS:
	case GR_NUM_FB:
	case GR_NUM_SWAP_HISTORY_BUFFER:
	case GR_NUM_TMU:
	case GR_PENDING_BUFFERSWAPS:
	case GR_REVISION_FB:
	case GR_REVISION_TMU:
	case GR_STATS_LINES:
	case GR_STATS_PIXELS_AFUNC_FAIL:
	case GR_STATS_PIXELS_CHROMA_FAIL:
	case GR_STATS_PIXELS_IN:
	case GR_STATS_PIXELS_OUT:
	case GR_STATS_POINTS:
	case GR_STATS_TRIANGLES_IN:
	case GR_STATS_TRIANGLES_OUT:
	case GR_SUPPORTS_PASSTHRU:
	case GR_TEXTURE_ALIGN:
		grGet((FxU32)pname, 4, quickArray);
		return PyInt_FromLong(quickArray[0]);
	case GR_FIFO_FULLNESS:
	case GR_VIDEO_POSITION:
	case GR_ZDEPTH_MIN_MAX:
	case GR_WDEPTH_MIN_MAX:
		grGet((FxU32)pname, 8, quickArray);
		return Py_BuildValue("(ll)", quickArray[0],
			quickArray[1]);
	case GR_VIEWPORT:
	case GR_BITS_RGBA:
		grGet((FxU32)pname, 16, quickArray);
		return Py_BuildValue("(llll)",
			quickArray[0], quickArray[1],
			quickArray[2], quickArray[3]);
	default:
		PyErr_SetString(PyExc_ValueError, "grGet param not recognised");
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grGetString(PyObject *self, PyObject *args)

{
	long mode;
	const char *res;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	res = grGetString((FxU32)mode);
	if (!res)
	{
		PyErr_SetString(PyExc_ValueError, "grGetString param unrecognised");
		return NULL;
	}
	return Py_BuildValue("s", res);
}

static PyObject *pg_grQueryResolutions(PyObject *self, PyObject *args)

{
	PyTupleObject *tuple;
	PyObject *resntuple;
	long res;
	long refresh;
	int numColorBuffers;
	int numAuxBuffers;
	int listSize;
	GrResolution query;
	GrResolution *response;
	int i;
	if (!PyArg_ParseTuple(args, "llll",
			&res, &refresh, &numColorBuffers,
			&numAuxBuffers))
			return NULL;
	query.resolution = (GrScreenResolution_t)res;
	query.refresh = (GrScreenRefresh_t)refresh;
	query.numColorBuffers = (int)numColorBuffers;
	query.numAuxBuffers = (int)numAuxBuffers;
	listSize = grQueryResolutions(&query, NULL) / sizeof(GrResolution);
	response = malloc(sizeof(GrResolution)*listSize);
	if (!response)
	{
		PyErr_SetString(PyExc_MemoryError, "grQueryResolutions");
		return NULL;
	}
	grQueryResolutions(&query, response);
	tuple = (PyTupleObject *)PyTuple_New(listSize);
	if (!tuple) {free(response); return NULL;}
	for(i=0; i<listSize; i++)
	{
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(tuple, i, Py_None);
	}
	for(i=0; i<listSize; i++)
	{
		resntuple = Py_BuildValue("(llll)",
			response[i].resolution,
			response[i].refresh,
			response[i].numColorBuffers,
			response[i].numAuxBuffers);
		if (!resntuple)
		{
			free(response); Py_DECREF(tuple);
			return NULL;
		}
		PyTuple_SET_ITEM(tuple, i, resntuple);
	}
	free(response);
	return (PyObject *)tuple;
}

static PyObject *pg_grReset(PyObject *self, PyObject *args)

{
	long what;
	FxBool ans;
	if (!PyArg_ParseTuple(args, "l", &what))
		return NULL;
	ans = grReset(what);
	return PyInt_FromLong((long)ans);
}

static PyObject *pg_grEnable(PyObject *self, PyObject *args)

{
	long mode;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	grEnable((GrEnableMode_t)mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDisable(PyObject *self, PyObject *args)

{
	long mode;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	grDisable((GrEnableMode_t)mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grCoordinateSpace(PyObject *self, PyObject *args)

{
	long mode;
	if (!PyArg_ParseTuple(args, "l", &mode))
		return NULL;
	grCoordinateSpace((GrCoordinateSpaceMode_t)mode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grDepthRange(PyObject *self, PyObject *args)

{
	double n, f;
	if (!PyArg_ParseTuple(args, "dd", &n, &f))
		return NULL;
	grDepthRange((FxFloat)n, (FxFloat)f);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grViewport(PyObject *self, PyObject *args)

{
	long x, y, width, height;
	if (!PyArg_ParseTuple(args, "llll", &x, &y,
		&width, &height))
		return NULL;
	grViewport(x, y, width, height);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexCalcMemRequired(PyObject *self, PyObject *args)

{
	long lodmin, lodmax, aspect, fmt;
	long size;
	if (!PyArg_ParseTuple(args, "llll", 
		&lodmin, &lodmax,
		&aspect, &fmt))
		return NULL;
	size = grTexCalcMemRequired((GrLOD_t)lodmin,
		(GrLOD_t)lodmax, (GrAspectRatio_t)aspect,
		(GrTextureFormat_t)fmt);

	return PyInt_FromLong(size);
}

static PyObject *pg_grTexTextureMemRequired(PyObject *self, PyObject *args)

{
	long evenOdd, lodmin, lodmax, aspect, fmt;
	long size;
	GrTexInfo info;
	if (!PyArg_ParseTuple(args, "lllll", &evenOdd,
		&lodmin, &lodmax,
		&aspect, &fmt))
		return NULL;
	info.smallLodLog2 = (GrLOD_t)lodmin;
	info.largeLodLog2 = (GrLOD_t)lodmax;
	info.aspectRatioLog2 = aspect;
	info.format = (GrTextureFormat_t)fmt;
	size = grTexTextureMemRequired((FxU32)evenOdd, &info);
	return PyInt_FromLong(size);
}

static PyObject *pg_grTexMinAddress(PyObject *self, PyObject *args)

{
	long tmu, addr;
	if (!PyArg_ParseTuple(args, "l", &tmu))
		return NULL;
	addr = grTexMinAddress(tmu);
	return PyInt_FromLong(addr);
}

static PyObject *pg_grTexMaxAddress(PyObject *self, PyObject *args)

{
	long tmu, addr;
	if (!PyArg_ParseTuple(args, "l", &tmu))
		return NULL;
	addr = grTexMaxAddress(tmu);
	return PyInt_FromLong(addr);
}

static PyObject *pg_grTexNCCTable(PyObject *self, PyObject *args)

{
	long table;
	if (!PyArg_ParseTuple(args, "l", &table))
		return NULL;
	grTexNCCTable((GrNCCTable_t)table);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexSource(PyObject *self, PyObject *args)

{
	long tmu, addr, evenodd, lodmin,
		lodmax, aspect, format;
	GrTexInfo info;
	if (!PyArg_ParseTuple(args, "lllllll",
		&tmu, &addr, &evenodd, &lodmin, &lodmax,
		&aspect, &format))
		return NULL;
	info.smallLodLog2 = (GrLOD_t)lodmin;
	info.largeLodLog2 = (GrLOD_t)lodmax;
	info.aspectRatioLog2 = aspect;
	info.format = (GrTextureFormat_t)format;
	grTexSource((GrChipID_t)tmu, addr, evenodd,
				&info);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexClampMode(PyObject *self, PyObject *args)

{
	long smode, tmode, tmu;
	if (!PyArg_ParseTuple(args, "lll",
			&tmu, &smode, &tmode))
		return NULL;
	grTexClampMode((GrChipID_t)tmu,
			       (GrTextureClampMode_t)smode,
				   (GrTextureClampMode_t)tmode);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexCombine(PyObject *self, PyObject *args)

{
	long tmu, rgbfun, rgbfact, alfun, alfact,
		 rgbinvert, alphainvert;
	if (!PyArg_ParseTuple(args, "lllllll",
		 &tmu, &rgbfun, &rgbfact, &alfun,
		 &alfact, &rgbinvert, &alphainvert))
		return NULL;
	grTexCombine((GrChipID_t)tmu,
		(GrCombineFunction_t)rgbfun,
		(GrCombineFactor_t)rgbfact,
		(GrCombineFunction_t)alfun,
		(GrCombineFactor_t)alfact,
		(FxBool)rgbinvert, (FxBool)alphainvert);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexDetailControl(PyObject *self, PyObject *args)

{
	long tmu, lodBias, detailScale;
	double detailMax;
	if (!PyArg_ParseTuple(args, "llld",
		 &tmu, &lodBias, &detailScale, &detailMax))
		return NULL;
	grTexDetailControl(
		(GrChipID_t)tmu, lodBias,
		(FxU8)detailScale, 
		(float)detailMax);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexFilterMode(PyObject *self, PyObject *args)

{
	long tmu, minMode, magMode;
	if (!PyArg_ParseTuple(args, "lll",
		 &tmu, &minMode, &magMode))
		return NULL;
	grTexFilterMode(
		(GrChipID_t)tmu,
		(GrTextureFilterMode_t)minMode,
		(GrTextureFilterMode_t)magMode);
		
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexLodBiasValue(PyObject *self, PyObject *args)

{
	double bias; long tmu;
	if (!PyArg_ParseTuple(args, "ld", &tmu, &bias))
		return NULL;
	grTexLodBiasValue((GrChipID_t)tmu, (float)bias);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexDownloadMipMap(PyObject *self, PyObject *args)

{
	long tmu, startAddr, evenOdd;
	long smallLod, largeLod, aspect, format;
	char *data;
	int datalen;
	GrTexInfo info;

	if (!PyArg_ParseTuple(args, "llllllls#",
		 &tmu, &startAddr, &evenOdd,
		 &smallLod, &largeLod, &aspect,
		 &format, &data, &datalen))
		return NULL;
	info.smallLodLog2 = (GrLOD_t)smallLod;	
	info.largeLodLog2 = (GrLOD_t)largeLod;
	info.aspectRatioLog2 = (GrAspectRatio_t)aspect;
	info.format = (GrTextureFormat_t)format;
	info.data = data;
	grTexDownloadMipMap((GrChipID_t)tmu,
		(FxU32)startAddr, (FxU32)evenOdd,
		&info);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexDownloadMipMapLevel(PyObject *self, PyObject *args)

{
	long tmu, startAddr;
	long thisLod, largeLod, aspect, format;
	long evenOdd;
	char *data;
	int datalen;

	if (!PyArg_ParseTuple(args, "llllllls#",
		 &tmu, &startAddr, &thisLod,
		 &largeLod, &aspect, &format, 
		 &evenOdd, &data, &datalen))
		return NULL;

	grTexDownloadMipMapLevel((GrChipID_t)tmu,
		(FxU32)startAddr, (GrLOD_t)thisLod,
		(GrLOD_t)largeLod,
		(GrAspectRatio_t)aspect,
		(GrTextureFormat_t)format,
		(FxU32)evenOdd, (void *)data);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexDownloadMipMapLevelPartial(PyObject *self, PyObject *args)

{
	long tmu, startAddr;
	long thisLod, largeLod, aspect, format;
	long evenOdd;
	char *data;
	int datalen;
	long start, end;
	long res;
	if (!PyArg_ParseTuple(args, "llllllls#ll",
		 &tmu, &startAddr, &thisLod,
		 &largeLod, &aspect, &format, 
		 &evenOdd, &data, &datalen,
		 &start, &end))
		return NULL;

	res = grTexDownloadMipMapLevelPartial(
		(GrChipID_t)tmu,
		(FxU32)startAddr, (GrLOD_t)thisLod,
		(GrLOD_t)largeLod,
		(GrAspectRatio_t)aspect,
		(GrTextureFormat_t)format,
		(FxU32)evenOdd, (void *)data,
		start, end);

	return PyInt_FromLong(res);
}


static int downloadNCCTable(int tableType, PyObject *seq)
{
	FxU32 inttable[52];
	GuNccTable table;
	int i;
	long j;
	for(i=0; i<52; i++)
	{
		if (!getLongFromSequence(&j, i, seq))
			return 0;
		inttable[i] = j;
	}
	for(i=0; i<16; i++)
		table.yRGB[i] = (FxU8)inttable[i];
	for(i=0; i<12; i++)
		table.iRGB[i/3][i%3] = (FxU16)(inttable[i+16]);
	for(i=0; i<12; i++)
		table.qRGB[i/3][i%3] = (FxU16)(inttable[i+28]);
	for(i=0; i<12; i++)
		table.packed_data[i] = (FxU32)(inttable[i+40]);
	grTexDownloadTable(tableType, &table);
	return 1;
}

static int downloadPalette(int tableType, PyObject *seq)
{
	GuTexPalette pal;
	int i;
	long j;
	for(i=0; i<256; i++)
	{
		if (!getLongFromSequence(&j, i, seq))
			return 0;
		pal.data[i] = j;
	}
	grTexDownloadTable(tableType, &pal);
	return 1;
}
			
	
static PyObject *pg_grTexDownloadTable(PyObject *self, PyObject *args)

{
	PyObject *dataList;
	long tableType;

	if (!PyArg_ParseTuple(args, "lO",
		 &tableType, &dataList))
		return NULL;
	switch (tableType)
	{
	case GR_TEXTABLE_NCC0:
	case GR_TEXTABLE_NCC1:
		if (!downloadNCCTable(tableType, dataList))
			return NULL;
		break;
	case GR_TEXTABLE_PALETTE:
	case GR_TEXTABLE_PALETTE_6666_EXT:
		if (!downloadPalette(tableType, dataList))
			return NULL;
		break;
	default:
		PyErr_SetString(PyExc_ValueError, "table type not known");
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexDownloadTablePartial(PyObject *self, PyObject *args)

{
	PyObject *dataList;
	long tableType;
	GuTexPalette pal;
	int i, q;
	long j;
	long start, end;
	if (!PyArg_ParseTuple(args, "lOll",
		 &tableType, &dataList,
		 &start, &end))
		return NULL;
	q = 0;
	for(i=start; i<=end; i++)
	{
		if (!getLongFromSequence(&j, q, dataList))
			return NULL;
		pal.data[i] = j;
		q++;
	}
	grTexDownloadTablePartial(tableType, &pal,
		start, end);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexMipMapMode(PyObject *self, PyObject *args)

{
	long tmu, mode, lodBlend;

	if (!PyArg_ParseTuple(args, "lll",
		 &tmu, &mode, &lodBlend))
		return NULL;
	grTexMipMapMode((GrChipID_t)tmu,
		(GrMipMapMode_t)mode,
		(FxBool)lodBlend);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexMultibase(PyObject *self, PyObject *args)

{
	long tmu, enable;
	if (!PyArg_ParseTuple(args, "lll",
		 &tmu, &enable))
		return NULL;
	grTexMultibase((GrChipID_t)tmu,
		(FxBool)enable);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grTexMultibaseAddress(PyObject *self, PyObject *args)

{
	long tmu, range;
	long startAddr, evenOdd;
	long minLod, maxLod, aspect, format;
	GrTexInfo info;

	/*"grTexMultibaseAddress(tmu,range,startAddr,evenOdd,minLOD,maxLOD,aspect,format)"*/

	if (!PyArg_ParseTuple(args, "llllllll",
		 &tmu, &range, &startAddr, &evenOdd, &minLod, &maxLod,
		 &aspect, &format))
		return NULL;
	info.smallLodLog2 = (GrLOD_t)minLod;
	info.largeLodLog2 = (GrLOD_t)maxLod;
	info.aspectRatioLog2 = (GrAspectRatio_t)aspect;
	info.format = (GrTextureFormat_t)format;

	grTexMultibaseAddress((GrChipID_t)tmu,
		(GrTexBaseRange_t)range,
		(FxU32)startAddr,
		(FxU32)evenOdd,
		&info);

	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject *pg_grLfbConstantAlpha(PyObject *self, PyObject *args)

{
	long alphaval;
	if (!PyArg_ParseTuple(args, "l",
		 &alphaval))
		return NULL;
	grLfbConstantAlpha((GrAlpha_t)alphaval);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grLfbConstantDepth(PyObject *self, PyObject *args)

{
	long depth;
	if (!PyArg_ParseTuple(args, "l",
		 &depth))
		return NULL;
	grLfbConstantDepth((FxU32)depth);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grLfbWriteRegion(PyObject *self, PyObject *args)

{
	long destBuf, destX, destY, srcFormat, srcW, srcH, pipeline,
		stride;
	char *data;
	long datalen;
	long ok;
	if (!PyArg_ParseTuple(args, "lllllllls#",
		&destBuf, &destX, &destY, &srcFormat, &srcW, &srcH,
		&pipeline, &stride, &data, &datalen))
		return NULL;
	ok = grLfbWriteRegion(destBuf, destX, destY, srcFormat,
		srcW, srcH, pipeline, stride, data);
	return PyInt_FromLong(ok);
}

static PyObject *pg_grLfbReadRegion(PyObject *self, PyObject *args)

{
	long srcBuf, srcx, srcy, srcw, srch,
		stride;
	char *data;
	long datalen;
	long ok;
	PyObject *string;
	if (!PyArg_ParseTuple(args, "llllll",
			&srcBuf, &srcx, &srcy, &srcw, &srch,
			&stride))
			return NULL;
	datalen = stride * srch * 2;
	data = malloc(datalen);
	if (!data)
	{
		PyErr_SetString(PyExc_MemoryError, "no read-buffer for region");
		return NULL;
	}
	ok = grLfbReadRegion(srcBuf, srcx, srcy, srcw, srch,
			stride, data);
	if (!ok)
	{
		free(data);
		Py_INCREF(Py_None);
		return Py_None;
	}
	string = PyString_FromStringAndSize(data, datalen);
	free(data);
	return string;
}

static PyObject *pg_grGlideSetState(PyObject *self, PyObject *args)

{
	char *s;
	int l;
	if (!PyArg_ParseTuple(args, "s#", &s, &l))
		return NULL;
	grGlideSetState(s);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grGlideGetState(PyObject *self, PyObject *args)

{
	char *s;
	FxI32 glideStateSize;
	PyObject *o;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	grGet(GR_GLIDE_STATE_SIZE, 4, &glideStateSize);
	s = malloc(glideStateSize);
	if (!s) {
		PyErr_SetString(PyExc_MemoryError, "getState memory failure");
		return NULL;
	}
	grGlideGetState(s);
	o = PyString_FromStringAndSize(s, glideStateSize);
	free(s);
	return o;
}

static PyObject *pg_grGlideSetVertexLayout(PyObject *self, PyObject *args)

{
	char *s;
	int l;
	if (!PyArg_ParseTuple(args, "s#", &s, &l))
		return NULL;
	grGlideSetVertexLayout(s+4);
	packedRGBOffset = *((FxI32 *)s);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pg_grGlideGetVertexLayout(PyObject *self, PyObject *args)
{
	char *s;
	FxI32 glideStateSize;
	PyObject *o;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	grGet(GR_GLIDE_VERTEXLAYOUT_SIZE, 4, &glideStateSize);
	glideStateSize += 4;
	s = malloc(glideStateSize);
	if (!s) {
		PyErr_SetString(PyExc_MemoryError, "getState memory failure");
		return NULL;
	}
	grGlideGetVertexLayout(s+4);
	*((FxI32 *)s) = packedRGBOffset;
	o = PyString_FromStringAndSize(s, glideStateSize);
	free(s);
	return o;
}

static PyMethodDef pyglide_methods[] = {
	{"grGlideInit", pg_grGlideInit, 1, "Initialises the Glide library"},
	{"grGlideShutdown", pg_grGlideShutdown, 1, "Shuts down the Glide library"},
	{"grSstWinOpen", pg_grSstWinOpen, 1,
		"ctxt = grSstWinOpen(hWnd,res,refresh,clrfmt,origin,nColBuf,nAuxBuf)"
	},
	{"grSstSelect", pg_grSstSelect, 1, "grSstSelect(sysnum)"},
	{"grSstWinClose", pg_grSstWinClose, 1, "ok = grSstWinClose(ctxt)"},
	{"grDrawPoint", pg_grDrawPoint, 1, "grDrawPoint(vertex)"},
	{"grDrawLine", pg_grDrawLine, 1, "grDrawLine(vertex1, vertex2)"},
	{"grDrawTriangle", pg_grDrawTriangle, 1, "grDrawTriangle(vertex1, vertex2, vertex3)"},
	{"grVertexLayout", pg_grVertexLayout, 1, "grVertexLayout(param, index, mode)"},
	{"grDrawVertexArray", pg_grDrawVertexArray, 1, "grDrawVertexArray(mode, vertexlist)"},
	{"grBufferClear", pg_grBufferClear, 1, "grBufferClear(color, alpha, depth)"},
	{"grBufferSwap", pg_grBufferSwap, 1, "grBufferSwap(swapInterval)"},
	{"grRenderBuffer", pg_grRenderBuffer, 1, "grRenderBuffer(buffer)"},
	{"grFinish", pg_grFinish, 1, "grFinish()"},
	{"grFlush", pg_grFlush, 1, "grFlush()"},
	{"grSelectContext", pg_grSelectContext, 1, "bool = grSelectContext(context)"},
	{"grSstOrigin", pg_grSstOrigin, 1, "grSstOrigin(origin)"},
	{"grAlphaBlendFunction", pg_grAlphaBlendFunction, 1,
		"grAlphaBlendFunction(rgb_sf, rgb_df, alpha_sf, alpha_df)"},
	{"grAlphaCombine", pg_grAlphaCombine, 1, 
		"grAlphaCombine(function, factor, local, other, invert)"},
	{"grAlphaControlsITRGBLighting", pg_grAlphaControlsITRGBLighting, 1,
		"grAlphaControlsITRGBLighting(enable)"},
	{"grAlphaTestFunction", pg_grAlphaTestFunction, 1, 
		"grAlphaTestFunction(function)"},
	{"grAlphaTestReferenceValue", pg_grAlphaTestReferenceValue, 1, 
		"grAlphaTestReferenceValue(value)"},
	{"grChromakeyMode", pg_grChromakeyMode, 1, "grChromakeyMode(mode)"},
	{"grChromakeyValue", pg_grChromakeyValue, 1, "grChromakeyValue(color)"},
	{"grClipWindow", pg_grClipWindow, 1, "grClipWindow(minx, miny, maxx, maxy)"},
	{"grColorCombine", pg_grColorCombine, 1,
		"grColorCombine(function, factor, local, other, invert)"},
	{"grColorMask", pg_grColorMask, 1,
		"grColorMask(rgb, a)"},
	{"grCullMode", pg_grCullMode, 1, "grCullMode(mode)"},
	{"grConstantColorValue", pg_grConstantColorValue, 1,
		"grConstantColorValue(color)"},
	{"grDepthBiasLevel", pg_grDepthBiasLevel, 1,
		"grDepthBiasLevel(level)"},
	{"grDepthBufferFunction", pg_grDepthBufferFunction, 1,
		"grDepthBufferFunction(function)"},
	{"grDepthBufferMode", pg_grDepthBufferMode, 1,
		"grDepthBufferMode(mode)"},
	{"grDepthMask", pg_grDepthMask, 1, "grDepthMask(mask)"},
	{"grDisableAllEffects", pg_grDisableAllEffects, 1,
		"grDisableAllEffects()"},
	{"grDitherMode", pg_grDitherMode, 1, "grDitherMode(mode)"},
	{"grFogColorValue", pg_grFogColorValue, 1, "grFogColorValue(fogcolor)"},
	{"grFogMode", pg_grFogMode, 1, "grFogMode(mode)"},
	{"grFogTable", pg_grFogTable, 1, "grFogTable(table)"},
	{"grLoadGammaTable", pg_grLoadGammaTable, 1,
		"grLoadGammaTable(redlist, greenlist, bluelist)"},
	{"grSplash", pg_grSplash, 1,
		"grSplash(x, y, width, height, frame)"},
	{"grGet", pg_grGet, 1, "value = grGet(pname)"},
	{"grGetString", pg_grGetString, 1, "string = grGetString(pname)"},
	{"grQueryResolutions", pg_grQueryResolutions, 1,
		"reslist = grQueryResolutions(res, refresh, colorBuffers, auxBuffers"},
	{"grReset", pg_grReset, 1, "bool = grReset(what)"},
	{"grEnable", pg_grEnable, 1, "grEnable(mode)"},
	{"grDisable", pg_grDisable, 1, "grDisable(mode)"},
	{"grCoordinateSpace", pg_grCoordinateSpace, 1, "grCoordinateSpace(mode)"},
	{"grDepthRange", pg_grDepthRange, 1, "grDepthRange(n, f)"},
	{"grViewport", pg_grViewport, 1, "grViewport(x, y, width, height)"},
	{"grTexCalcMemRequired", pg_grTexCalcMemRequired, 1,
		"size = grTexCalcMemRequired(lodmin, lodmax, aspect, format)"},
	{"grTexTextureMemRequired", pg_grTexTextureMemRequired, 1,
		"size=grTexTextureMemRequired(evenOdd,lodmin,lodmax,aspect,format"},
	{"grTexMinAddress", pg_grTexMinAddress, 1, "addr = grTexMinAddress(tmu)"},
	{"grTexMaxAddress", pg_grTexMaxAddress, 1, "addr = grTexMaxAddress(tmu)"},
	{"grTexNCCTable", pg_grTexNCCTable, 1, "grTexNCCTable(table)"},
	{"grTexSource", pg_grTexSource, 1, 
		"grTexSource(tmu,startAddr,evenodd,lodmin,lodmax,aspect,format"},
	{"grTexClampMode", pg_grTexClampMode, 1,
		"grTexClampMode(tmu, s_clampmode, t_clampmode)"},
	{"grTexCombine", pg_grTexCombine, 1,
	"grTexCombine(tmu,rgbFn,rgbFact,alphaFn,alphaFact,rgbInvert,alphaInvert"},
	{"grTexDetailControl", pg_grTexDetailControl, 1,
		"grTexDetailControl(tmu, lod_bias, detail_scale, detail_max)"},
	{"grTexFilterMode", pg_grTexFilterMode, 1,
		"grTexFilterMode(tmu, minfilter_mode, magfilter_mode)"},
	{"grTexLodBiasValue", pg_grTexLodBiasValue, 1,
		"grTexLodBiasValue(tmu, bias)"},
	{"grTexDownloadMipMap", pg_grTexDownloadMipMap, 1,
		"grTexDownloadMipMap(tmu,startAddress,evenOdd,smallLod,largeLod,aspect,fmt,data)"},
	{"grTexDownloadMipMapLevel", pg_grTexDownloadMipMapLevel, 1,
	"grTexDownloadMipMapLevel(tmu,addr,thisLOD,largeLOD,aspect,format,evenOdd,data)"},
	{"grTexDownloadMipMapLevelPartial", pg_grTexDownloadMipMapLevelPartial, 1,
	"(tmu,addr,thisLOD,largeLOD,aspectRatio,format,evenOdd,data,start,end)"},
	{"grTexDownloadTable", pg_grTexDownloadTable, 1,
		"grTexDownloadTable(type, data)"},
	{"grTexDownloadTablePartial", pg_grTexDownloadTablePartial, 1,
		"grTexDownloadTablePartial(type, data, start, end)"},
	{"grTexMipMapMode", pg_grTexMipMapMode, 1,
		"grTexMipMapMode(tmu, mode, lodBlend)"},
	{"grTexMultibase", pg_grTexMultibase, 1,
		"grTexMultibase(tmu, enable)"},
	{"grTexMultibaseAddress", pg_grTexMultibaseAddress, 1,
	"grTexMultibaseAddress(tmu,range,startAddr,evenOdd,minLOD,maxLOD,aspect,format)"},

	{"grGlideGetState", pg_grGlideGetState, 1,
		"string = grGlideGetState()"},
	{"grGlideSetState", pg_grGlideSetState, 1,
		"grGlideSetState(string)"},
	{"grGlideGetVertexLayout", pg_grGlideGetVertexLayout, 1,
		"string = grGlideGetVertexLayout()"},
	{"grGlideSetVertexLayout", pg_grGlideSetVertexLayout, 1,
		"grGlideSetVertexLayout(string)"},
	{"grLfbConstantAlpha", pg_grLfbConstantAlpha, 1,
		"grLfbConstantAlpha(alpha)"},
	{"grLfbConstantDepth", pg_grLfbConstantDepth, 1,
		"grLfbConstantDepth(depth)"},
	{"grLfbWriteRegion", pg_grLfbWriteRegion, 1,
	"ok=grLfbWriteRegion(dstBuf,dstX,dstY,srcFmt,srcW,srcH,pipeline,stride,data)"},
	{"grLfbReadRegion", pg_grLfbReadRegion, 1,
	 "data|None=grLfbReadRegion(buffer,srcX,srcY,srcW,srcH,destStride)"},

	{NULL, NULL} /* SENTINEL */
};

void
initpyglide()
{
	Py_InitModule("pyglide", pyglide_methods);
	vertArray = fixedArray;
	vertSize = 5;
	packedRGBOffset = -1;
}